<?php
/**
 * @package      ui/Theme-Builder Lite
 * @author       Stephan W.
 * @author       url   https://www.ui-themebuilder.com/
 * @copyright    Copyright (C) 2021 ui-themebuilder.com, All rights reserved.
 * @developer    Stephan Wittling - https://www.ui-themebuilder.com/
 *               ui/Theme-Builder Lite is distributed in the hope that it will be useful,
 *               but WITHOUT ANY WARRANTY; without even the implied warranty of
 *               MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 *               See the GNU General Public License for more details.
 * @license      http://www.gnu.org/licenses/gpl.html GNU/GPL
 *********************************************************************************/

namespace SW\Component\uiThemeBuilderLite\Site\Service;

defined('_JEXEC') or die;

use Joomla\CMS\Application\SiteApplication;
use Joomla\CMS\Categories\CategoryFactoryInterface;
use Joomla\CMS\Categories\CategoryInterface;
use Joomla\CMS\Component\ComponentHelper;
use Joomla\CMS\Component\Router\RouterView;
use Joomla\CMS\Component\Router\RouterViewConfiguration;
use Joomla\CMS\Component\Router\Rules\MenuRules;
use Joomla\CMS\Component\Router\Rules\NomenuRules;
use Joomla\CMS\Component\Router\Rules\StandardRules;
use Joomla\CMS\Menu\AbstractMenu;
use Joomla\Database\DatabaseInterface;
use Joomla\Database\ParameterType;

/**
 * Routing class from com_uithemebuilderlite
 *
 * @since  v1.0.0
 */
class Router extends RouterView
{
    /**
     * Flag to remove IDs
     *
     * @var    boolean
     *
     * @since   v1.0.0
     */
    protected bool $noIDs = false;

    /**
     * The category factory
     *
     * @var CategoryFactoryInterface
     *
     * @since  1.0.0
     */
    private CategoryFactoryInterface $categoryFactory;

    /**
     * The category cache
     *
     * @var  array
     *
     * @since  1.0.0
     */
    private array $categoryCache = [];

    /**
     * The db
     *
     * @var DatabaseInterface
     *
     * @since  1.0.0
     */
    private DatabaseInterface $db;

    /**
     * Content Component router constructor
     *
     * @param SiteApplication          $app             The application object
     * @param AbstractMenu             $menu            The menu object to work with
     * @param CategoryFactoryInterface $categoryFactory The category object
     * @param DatabaseInterface        $db              The database object
     *
     * @since   v1.0.0
     */
    public function __construct(SiteApplication $app, AbstractMenu $menu, CategoryFactoryInterface $categoryFactory, DatabaseInterface $db)
    {
        $this->categoryFactory = $categoryFactory;
        $this->db = $db;

        $params = ComponentHelper::getParams('com_content');
        $this->noIDs = (bool)$params->get('sef_ids');
        $categories = new RouterViewConfiguration('categories');
        $categories->setKey('id');
        $this->registerView($categories);
        $category = new RouterViewConfiguration('category');
        $category->setKey('id')->setParent($categories, 'catid')->setNestable();
        $this->registerView($category);
        $article = new RouterViewConfiguration('page');
        $article->setKey('id')->setParent($category, 'catid');
        $this->registerView($article);
        $this->registerView(new RouterViewConfiguration('archive'));
        $this->registerView(new RouterViewConfiguration('featured'));
        $form = new RouterViewConfiguration('form');
        $form->setKey('a_id');
        $this->registerView($form);

        parent::__construct($app, $menu);

        $this->attachRule(new MenuRules($this));
        $this->attachRule(new StandardRules($this));
        $this->attachRule(new NomenuRules($this));
    }

    /**
     * Method to get the segment(s) for a category
     *
     * @param string $id    ID of the category to retrieve the segments for
     * @param array  $query The request that is built right now
     *
     * @return  array|string  The segments of this item
     *
     * @since   v1.0.0
     */
    public function getCategoriesSegment($id, $query): array|string
    {
        return $this->getCategorySegment($id, $query);
    }

    /**
     * Method to get the segment(s) for a category
     *
     * @param   string  $id     ID of the category to retrieve the segments for
     * @param   array   $query  The request that is built right now
     *
     * @return  array|string  The segments of this item
     *
     * @since   v1.0.0
     */
    public function getCategorySegment(string $id, array $query): array|string
    {
        $category = $this->getCategories(['access' => true])->get($id);

        if ($category) {
            $path = array_reverse($category->getPath(), true);
            $path[0] = '1:root';

            if ($this->noIDs) {
                foreach ($path as &$segment) {
                    list($id, $segment) = explode(':', $segment, 2);
                }
            }

            return $path;
        }

        return array();
    }

    /**
     * Method to get categories from cache
     *
     * @param array $options The options for retrieving categories
     *
     * @return  CategoryInterface  The object containing categories
     *
     * @since   1.0.0
     */
    private function getCategories(array $options = []): CategoryInterface
    {
        $key = serialize($options);

        if (!isset($this->categoryCache[$key])) {
            $this->categoryCache[$key] = $this->categoryFactory->createCategory($options);
        }

        return $this->categoryCache[$key];
    }

    /**
     * Method to get the segment(s) for a form
     *
     * @param string $id    ID of the article form to retrieve the segments for
     * @param array  $query The request that is built right now
     *
     * @return  array|string  The segments of this item
     *
     * @since   v1.0.0
     */
    public function getFormSegment($id, $query): array|string
    {
        return $this->getPageSegment($id, $query);
    }

    /**
     * Method to get the segment(s) for an article
     *
     * @param string $id    ID of the article to retrieve the segments for
     * @param array  $query The request that is built right now
     *
     * @return  array|string  The segments of this item
     *
     * @since   v1.0.0
     */
    public function getPageSegment($id, $query): array|string
    {
        if (!strpos($id, ':')) {
            $id = (int)$id;
            $dbquery = $this->db->getQuery(true);
            $dbquery->select($this->db->quoteName('alias'))
                ->from($this->db->quoteName('#__uithemebuilderlite'))
                ->where($this->db->quoteName('id') . ' = :id')
                ->bind(':id', $id, ParameterType::INTEGER);
            $this->db->setQuery($dbquery);

            $id .= ':' . $this->db->loadResult();
        }

        if ($this->noIDs) {
            list($void, $segment) = explode(':', $id, 2);

            return array($void => $segment);
        }

        return array((int)$id => $id);
    }

    /**
     * Method to get the segment(s) for a category
     *
     * @param string $segment Segment to retrieve the ID for
     * @param array  $query   The request that is parsed right now
     *
     * @return  mixed   The id of this item or false
     *
     * @since   v1.0.0
     */
    public function getCategoriesId($segment, $query): mixed
    {
        return $this->getCategoryId($segment, $query);
    }

    /**
     * Method to get the id for a category
     *
     * @param string $segment Segment to retrieve the ID for
     * @param array  $query   The request that is parsed right now
     *
     * @return  mixed   The id of this item or false
     *
     * @since   v1.0.0
     */
    public function getCategoryId($segment, $query): mixed
    {
        if (isset($query['id'])) {
            $category = $this->getCategories(['access' => false])->get($query['id']);

            if ($category) {
                foreach ($category->getChildren() as $child) {
                    if ($this->noIDs) {
                        if ($child->alias == $segment) {
                            return $child->id;
                        }
                    } else {
                        if ($child->id == (int)$segment) {
                            return $child->id;
                        }
                    }
                }
            }
        }

        return false;
    }

    /**
     * Method to get the segment(s) for an page
     *
     * @param string $segment Segment of the page to retrieve the ID for
     * @param array  $query   The request that is parsed right now
     *
     * @return  mixed   The id of this item or false
     *
     * @since   v1.0.0
     */
    public function getPageId($segment, $query): mixed
    {
        if ($this->noIDs) {
            $dbquery = $this->db->getQuery(true);
            $dbquery->select($this->db->quoteName('id'))
                ->from($this->db->quoteName('#__uithemebuilderlite'))
                ->where(
                    [
                        $this->db->quoteName('alias') . ' = :alias',
                        $this->db->quoteName('catid') . ' = :catid',
                    ]
                )
                ->bind(':alias', $segment)
                ->bind(':catid', $query['id'], ParameterType::INTEGER);
            $this->db->setQuery($dbquery);

            return (int)$this->db->loadResult();
        }

        return (int)$segment;
    }
}
