<?php
/**
 * @package      ui/Theme-Builder Lite
 * @author       Stephan W.
 * @author url   https://www.ui-themebuilder.com/
 * @copyright Copyright (C) 2021 ui-themebuilder.com, All rights reserved.
 * @developer Stephan Wittling - https://www.ui-themebuilder.com/

 *               ui/Theme-Builder Lite is distributed in the hope that it will be useful,
 *               but WITHOUT ANY WARRANTY; without even the implied warranty of
 *               MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 *               See the GNU General Public License for more details.

 * @license      http://www.gnu.org/licenses/gpl.html GNU/GPL
 *********************************************************************************/
namespace SW\Component\uiThemeBuilderLite\Site\Model;

defined('_JEXEC') or die;

use Joomla\CMS\Component\ComponentHelper;
use Joomla\CMS\Factory;
use Joomla\CMS\Language\Multilanguage;
use Joomla\CMS\MVC\Model\ListModel;
use Joomla\Database\ParameterType;

/**
 * Featured ui/Theme-Builder Lite model class.
 *
 * @since  v1.0.0
 */
class FeaturedModel extends ListModel
{
    /**
     * Constructor.
     *
     * @param   array  $config  An optional associative array of configuration settings.
     *
     * @throws \Exception
     * @since   v1.0.0
     */
    public function __construct($config = [])
    {
        if (empty($config['filter_fields'])) {
            $config['filter_fields'] = [
                'id', 'a.id',
                'title', 'a.title',
                'hits', 'a.hits',
                'published', 'a.published',
                'ordering', 'a.ordering',
            ];
        }

        parent::__construct($config);
    }

    /**
     * Method to get a list of items.
     *
     * @return  mixed  An array of objects on success, false on failure.
     * @since   v1.0.0
     */
    public function getItems()
    {
        // Invoke the parent getItems method to get the main list
        $items = parent::getItems();

        // Convert the params field into an object, saving original in _params
        for ($i = 0, $n = count($items); $i < $n; $i++) {
            $item = &$items[$i];

            /*          if (!isset($this->_params)) {
                            $item->params = new Registry($item->params);
                        } */
        }

        return $items;
    }

    /**
     * Method to build an SQL query to load the list data.
     *
     *
     * @since   v1.0.0
     */
    protected function getListQuery(): \Joomla\Database\DatabaseQuery
    {
        $user = Factory::getApplication()->getIdentity();
        $groups = $user->getAuthorisedViewLevels();

        // Create a new query object.
        $db    = Factory::getContainer()->get('DatabaseDriver');
        $query = $db->getQuery(true);

        // Select required fields from the categories.
        $query->select($this->getState('list.select', 'a.*'))
            ->from($db->quoteName('#__uithemebuilderlite', 'a'))
            ->where($db->quoteName('a.featured') . ' = 1')
            ->whereIn($db->quoteName('a.access'), $groups)
            ->innerJoin($db->quoteName('#__categories', 'c') . ' ON c.id = a.catid')
            ->whereIn($db->quoteName('c.access'), $groups);

        // Filter by category.
        if ($categoryId = $this->getState('category.id')) {
            $query->where($db->quoteName('a.catid') . ' = :catid');
            $query->bind(':catid', $categoryId, ParameterType::INTEGER);
        }

        $query->select('c.published as cat_published, c.published AS parents_published')
            ->where('c.published = 1');

        // Filter by state
        $state = $this->getState('filter.published');

        if (is_numeric($state)) {
            $query->where($db->quoteName('a.published') . ' = :published');
            $query->bind(':published', $state, ParameterType::INTEGER);

            // Filter by start and end dates.
            $nowDate = Factory::getDate()->toSql();

            $query->where('(' . $db->quoteName('a.publish_up') .
                ' IS NULL OR ' . $db->quoteName('a.publish_up') . ' <= :publish_up)')
                ->where('(' . $db->quoteName('a.publish_down') .
                    ' IS NULL OR ' . $db->quoteName('a.publish_down') . ' >= :publish_down)')
                ->bind(':publish_up', $nowDate)
                ->bind(':publish_down', $nowDate);
        }

        // Filter by language
        if ($this->getState('filter.language')) {
            $language = [Factory::getApplication()->getLanguage()->getTag(), '*'];
            $query->whereIn($db->quoteName('a.language'), $language, ParameterType::STRING);
        }

        // Add the list ordering clause.
        $query->order($db->escape($this->getState('list.ordering', 'a.ordering')) . ' ' . $db->escape($this->getState('list.direction', 'ASC')));

        return $query;
    }

    /**
     * Method to auto-populate the model state.
     *
     * Note. Calling getState in this method will result in recursion.
     *
     * @param   string  $ordering   An optional ordering field.
     * @param   string  $direction  An optional direction (asc|desc).
     *
     * @return  void
     *
     * @since   v1.0.0
     */
    protected function populateState($ordering = null, $direction = null): void
    {
        $app = Factory::getApplication();
        $params = ComponentHelper::getParams('com_uithemebuilderlite');

        // List state information
        $limit = $app->getUserStateFromRequest('global.list.limit', 'limit', $app->get('list_limit'), 'uint');
        $this->setState('list.limit', $limit);

        $limitstart = $app->input->get('limitstart', 0, 'uint');
        $this->setState('list.start', $limitstart);

        $orderCol = $app->input->get('filter_order', 'ordering');

        if (!in_array($orderCol, $this->filter_fields)) {
            $orderCol = 'ordering';
        }

        $this->setState('list.ordering', $orderCol);

        $listOrder = $app->input->get('filter_order_Dir', 'ASC');

        if (!in_array(strtoupper($listOrder), ['ASC', 'DESC', ''])) {
            $listOrder = 'ASC';
        }

        $this->setState('list.direction', $listOrder);

        $user = Factory::getApplication()->getIdentity();

        if ((!$user->authorise('core.edit.state', 'com_uithemebuilderlite')) && (!$user->authorise('core.edit', 'com_uithemebuilderlite'))) {
            // Limit to published for people who can't edit or edit.state.
            $this->setState('filter.published', 1);

            // Filter by start and end dates.
            $this->setState('filter.publish_date', true);
        }

        $this->setState('filter.language', Multilanguage::isEnabled());

        // Load the parameters.
        $this->setState('params', $params);
    }
}
