<?php
/**
* @package      ui/Theme-Builder Lite
* @author       Stephan W.
* @author url   https://www.ui-themebuilder.com/
* @copyright Copyright (C) 2021 ui-themebuilder.com, All rights reserved.
* @developer Stephan Wittling - https://www.ui-themebuilder.com/

*               ui/Theme-Builder Lite is distributed in the hope that it will be useful,
*               but WITHOUT ANY WARRANTY; without even the implied warranty of
*               MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
*               See the GNU General Public License for more details.

* @license      http://www.gnu.org/licenses/gpl.html GNU/GPL
*********************************************************************************/
namespace SW\Component\uiThemeBuilderLite\Site\Model;

defined('_JEXEC') or die;

use Joomla\CMS\Categories\Categories;
use Joomla\CMS\Categories\CategoryNode;
use Joomla\CMS\Factory;
use Joomla\CMS\Helper\TagsHelper;
use Joomla\CMS\Language\Multilanguage;
use Joomla\CMS\MVC\Model\ListModel;
use Joomla\CMS\Table\Table;
use Joomla\Database\DatabaseQuery;
use Joomla\Database\ParameterType;
use Joomla\Database\QueryInterface;
use Joomla\Registry\Registry;
use Joomla\CMS\Language\Associations;
use Joomla\Utilities\ArrayHelper;
use SW\Component\uiThemeBuilderLite\Site\Helper\AssociationHelper;
use SW\Component\uiThemeBuilderLite\Site\Helper\QueryHelper;

/**
 * This models supports retrieving a category, the pages associated with the category,
 * sibling, child and parent categories.
 *
 * @since  1.0.0
 */
class CategoryModel extends ListModel
{
    /**
     * Category items data
     *
     * @var  array
     *
     * @since  1.0.0
     */
    protected $_item = null;

    /**
     * Category left and right of this one
     *
     * @var  CategoryNode[]|null
     *
     * @since  1.0.0
     */
    protected $_siblings = null;

    /**
     * Array of child-categories
     *
     * @var  CategoryNode[]|null
     *
     * @since  1.0.0
     */
    protected $_children = null;

    /**
     * Parent category of the current one
     *
     * @var  CategoryNode|null
     *
     * @since  1.0.0
     */
    protected $_parent = null;

    /**
     * Model context string.
     *
     * @var  string
     *
     * @since  1.0.0
     */
    protected $_context = 'com_uithemebuilderlite.category';

    /**
     * The category that applies.
     *
     * @var  object
     *
     * @since  1.0.0
     */
    protected $_category = null;

    /**
     * The list of categories.
     *
     * @var  array
     *
     * @since  1.0.0
     */
    protected $_categories = null;


    /**
     * Constructor.
     *
     * @param   array  $config  An optional associative array of configuration settings.
     *
     * @throws \Exception
     * @since   v1.0.0
     */
    public function __construct($config = [])
    {
        if (empty($config['filter_fields'])) {
            $config['filter_fields'] = array(
                'id', 'a.id',
                'title', 'a.title',
                'alias', 'a.alias',
                'catid', 'a.catid', 'category_title',
                'state', 'a.state',
                'access', 'a.access', 'access_level',
                'created', 'a.created',
                'created_by', 'a.created_by',
                'modified', 'a.modified',
                'ordering', 'a.ordering',
                'featured', 'a.featured',
                'language', 'a.language',
                'hits', 'a.hits',
                'publish_up', 'a.publish_up',
                'publish_down', 'a.publish_down',
                'author', 'a.author',
                'filter_tag'
            );
        }
        parent::__construct($config);
    }


    /**
     * Method to auto-populate the model state.
     *
     * Note. Calling getState in this method will result in recursion.
     *
     * @param   string  $ordering   An optional ordering field.
     * @param   string  $direction  An optional direction (asc|desc).
     *
     * @return  void
     *
     * @throws \Exception
     * @since   v1.0.0
     */
    protected function populateState($ordering = null, $direction = null): void
    {
        $app = Factory::getApplication();
        $pk  = $app->input->getInt('id');

        $this->setState('category.id', $pk);

        // Load the parameters. Merge Global and Menu Item params into new object
        $params = $app->getParams();

        if ($menu = $app->getMenu()->getActive()) {
            $menuParams = $menu->getParams();
        } else {
            $menuParams = new Registry();
        }

        $mergedParams = clone $menuParams;
        $mergedParams->merge($params);

        $this->setState('params', $mergedParams);
        $user  = Factory::getApplication()->getIdentity();

        $asset = 'com_uithemebuilderlite';

        if ($pk) {
            $asset .= '.category.' . $pk;
        }

        if ((!$user->authorise('core.edit.state', $asset)) &&  (!$user->authorise('core.edit', $asset))) {
            // Limit to published for people who can't edit or edit.state.
            $this->setState('filter.published', 1);
        } else {
            $this->setState('filter.published', [0, 1]);
        }

        // Process show_noauth parameter
        if (!$params->get('show_noauth')) {
            $this->setState('filter.access', true);
        } else {
            $this->setState('filter.access', false);
        }

        $itemid = $app->input->get('id', 0, 'int') . ':' . $app->input->get('Itemid', 0, 'int');

        $value = $this->getUserStateFromRequest('com_uithemebuilderlite.category.filter.' . $itemid . '.tag', 'filter_tag', 0, 'int', false);
        $this->setState('filter.tag', $value);

        // Optional filter text
        $search = $app->getUserStateFromRequest('com_uithemebuilderlite.category.list.' . $itemid . '.filter-search', 'filter-search', '', 'string');
        $this->setState('list.filter', $search);


        $listOrder = $app->getUserStateFromRequest('com_uithemebuilderlite.category.list.' . $itemid . '.filter_order_Dir', 'filter_order_Dir', '', 'cmd');
        if (!in_array(strtoupper($listOrder), array('ASC', 'DESC', ''))) {
            $listOrder = 'ASC';
        }

        $this->setState('list.direction', $listOrder);


        // List state information
        $format = $app->input->getWord('format');
        $numOfPagesToDisplay = $mergedParams->get('display_num');
        if ($format === 'feed') {
            $limit = $app->get('feed_limit');
        } elseif (isset($numOfPagesToDisplay)) {
            $limit = $numOfPagesToDisplay;
        } else {
            $limit = 1;
        }
        $this->setState('list.limit', $limit);
        $limitstart = $app->input->get('limitstart', 0, 'uint');
        $this->setState('list.start', $limitstart);

        // Optional filter text
        // Set the depth of the category query based on parameter
        $showSubcategories = $params->get('show_subcategory_content', '0');
        if ($showSubcategories) {
            $this->setState('filter.max_category_levels', $params->get('show_subcategory_content', '1'));
            $this->setState('filter.subcategories', true);
        }

        $this->setState('filter.language', Multilanguage::isEnabled());

        $this->setState('layout', $app->input->getString('layout'));

        // Set the featured articles state
        $this->setState('filter.featured', $params->get('show_featured'));
    }


    /**
     * Method to get a list of items.
     *
     * @return  mixed  An array of objects on success, false on failure.
     * @throws \Exception
     *
     * @since  1.0.0
     */
    public function getItems()
    {

        $user  = Factory::getApplication()->getIdentity();
        $userId = $user->get('id');
        $guest  = $user->get('guest');
        $groups = $user->getAuthorisedViewLevels();

        // Invoke the parent getItems method to get the main list
        $items = parent::getItems();
        if ($items === false) {
            return false;
        }
        // Convert the params field into an object, saving original in _params
        for ($i = 0, $n = count($items); $i < $n; $i++) {
            $item = &$items[$i];

            // if (!isset($this->_params)) {
                // $item->params = new Registry($item->params);
                $item->params = clone $this->getState('params');
            // }

            // Some contexts may not use tags data at all, so we allow callers to disable loading tag data
            if ($this->getState('load_tags', true)) {
                $this->tags = new TagsHelper;
                $this->tags->getItemTags('com_uithemebuilderlite.page', $item->id);
            }

            if (Associations::isEnabled() && $item->params->get('show_associations')) {
                $item->associations = AssociationHelper::displayAssociations($item->id);
            }

            // Get display date
            switch (true) {
                case 'modified':
                    $item->displayDate = $item->modified;
                    break;

                case 'published':
                    $item->displayDate = ($item->publish_up === "0") ? $item->created : $item->publish_up;
                    break;

                default:
                case 'created':
                    $item->displayDate = $item->created;
                    break;
            }

            //
            //Compute the asset access permissions.
            //Technically guest could edit an page, but lets not check that to improve performance a little.
            if (!$guest) {
                $asset = 'com_uithemebuilderlite.page.' . $item->id;

                // Check general edit permission first.
                if ($user->authorise('core.edit', $asset)) {
                    $item->params->set('access-edit', true);
                }

                // Now check if edit.own is available.
                elseif (!empty($userId) && $user->authorise('core.edit.own', $asset)) {
                    // Check for a valid user and that they are the owner.
                    if ($userId == $item->created_by) {
                        $item->params->set('access-edit', true);
                    }
                }
            }

            $access = $this->getState('filter.access');
            if ($access) {
                // If the access filter has been set, we already have only the articles this user can view.
                $item->params->set('access-view', true);
            } else {
                // If no access filter is set, the layout takes some responsibility for display of limited information.
                if ($item->catid === "0") {
                    $item->params->set('access-view', in_array($item->access, $groups, true));
                } else {
                    $item->params->set('access-view', in_array($item->access, $groups, true));
                }
            }
        }

        return $items;
    }

    /**
     * Method to build an SQL query to load the list data.
     *
     * @return  \Joomla\Database\QueryInterface    An SQL query
     *
     * @throws \Exception
     * @since   v1.0.0
     */
    protected function getListQuery(): QueryInterface
    {

        $user   = Factory::getApplication()->getIdentity();
        $groups = $user->getAuthorisedViewLevels();

        // Create a new query object.
        $db    = Factory::getContainer()->get('DatabaseDriver');
        $query = $db->getQuery(true);
        $query->select($this->getState('list.select', 'a.*'))
            ->select($this->getSlugColumn($query, 'a.id', 'a.alias') . ' AS slug')
            ->select($this->getSlugColumn($query, 'c.id', 'c.alias') . ' AS catslug')
            ->from($db->quoteName('#__uithemebuilderlite', 'a'))
            ->leftJoin($db->quoteName('#__categories', 'c') . ' ON c.id = a.catid')
            ->whereIn($db->quoteName('a.access'), $groups);

        // Filter by category.
        if ($categoryId = $this->getState('category.id')) {
            $query->where($db->quoteName('a.catid') . ' = :acatid')
                ->whereIn($db->quoteName('c.access'), $groups);
            $query->bind(':acatid', $categoryId, ParameterType::INTEGER);
        }

        // Filter by state
        $state = $this->getState('filter.published');
        if (is_numeric($state)) {
            $query->where($db->quoteName('a.published') . ' = :published');
            $query->bind(':published', $state, ParameterType::INTEGER);
        } else {
            $query->whereIn($db->quoteName('c.published'), [0,1,2]);
        }

        // Filter by start and end dates.
        $nowDate = Factory::getDate()->toSql();
        if ($this->getState('filter.publish_date')) {
            $query->where('(' . $db->quoteName('a.publish_up')
                . ' IS NULL OR ' . $db->quoteName('a.publish_up') . ' <= :publish_up)')
                ->where('(' . $db->quoteName('a.publish_down')
                    . ' IS NULL OR ' . $db->quoteName('a.publish_down') . ' >= :publish_down)')
                ->bind(':publish_up', $nowDate)
                ->bind(':publish_down', $nowDate);
        }

        // Filter by search in title
        $search = $this->getState('list.filter');
        if (!empty($search)) {
            $search = '%' . trim($search) . '%';
            $query->where($db->quoteName('a.title') . ' LIKE :title ');
            $query->bind(':title', $search);
        }

        // Filter on the language.
        if ($language = $this->getState('filter.language')) {
            $language = [Factory::getLanguage()->getTag(), '*'];
            $query->whereIn($db->quoteName('a.language'), $language);
        }

        // Filter.order
        $query->order($this->getState('list.ordering', $this->_buildContentOrderBy()));


        // Filter by featured state
        $featured = $this->getState('filter.featured');
        switch ($featured) {
            case 'hide':
                $query->where($db->quoteName('a.featured') . ' = 0');
                break;

            case 'only':
                $query->where($db->quoteName('a.featured') . ' = 1');
                break;

            case 'show':
            default:
                // Normally we do not discriminate between featured/unfeatured items.
                break;
        }

        return $query;
    }

    /**
     * Build the orderby for the query
     *
     * @return  string  $orderby portion of query
     *
     * @throws \Exception
     *
     * @since   1.5
     */
    protected function _buildContentOrderBy()
    {
        $app       = Factory::getApplication();
        $db        = $this->getDatabase();
        $params    = $this->state->params;
        $itemid    = $app->input->get('id', 0, 'int') . ':' . $app->input->get('Itemid', 0, 'int');
        $orderCol  = $app->getUserStateFromRequest('com_uithemebuilderlite.category.list.' . $itemid . '.filter_order', 'filter_order', '', 'string');
        $orderDirn = $app->getUserStateFromRequest('com_uithemebuilderlite.category.list.' . $itemid . '.filter_order_Dir', 'filter_order_Dir', '', 'cmd');
        $orderby   = ' ';

        if (!in_array(strtoupper($orderDirn), array('ASC', 'DESC', ''))) {
            $orderDirn = 'ASC';
        }

        if ($orderCol && $orderDirn) {
            $orderby .= $db->escape($orderCol) . ' ' . $db->escape($orderDirn) . ', ';
        }

        $pageOrderby   = $params->get('orderby_sec', 'rdate');
        $pageOrderDate = $params->get('order_date');
        $categoryOrderby  = $params->def('orderby_pri', '');
        $secondary        = QueryHelper::orderbySecondary($pageOrderby, $pageOrderDate, $this->getDatabase()) . ', ';
        $primary          = QueryHelper::orderbyPrimary($categoryOrderby);

        $orderby .= $primary . ' ' . $secondary . ' a.created ';

        return $orderby;
    }

    /**
     * Method to get category data for the current category
     *
     * @return  object  The category object
     *
     * @throws \Exception
     * @since   v1.0.0
     */
    public function getCategory(): object|array
    {
        if (!is_object($this->_item)) {
            $app = Factory::getApplication();
            $menu = $app->getMenu();
            $active = $menu->getActive();
            if ($active) {
                $params = $active->getParams();
            } else {
                $params = new Registry;
            }
            if (isset($this->state->params)) {
                $params = $this->state->params;
                $options = array();
                $options['countItems'] = $params->get('show_cat_num_pages', 1) || !$params->get('show_empty_categories_cat', 0);
                $options['access']     = $params->get('check_access_rights', 1);
            } else {
                $options['countItems'] = 0;
            }

            $categories = Categories::getInstance('uiThemeBuilderLite', $options);
            $this->_item = $categories->get($this->getState('category.id', 'root'));
            // Compute selected asset permissions.
            if (is_object($this->_item)) {
                $user  = Factory::getApplication()->getIdentity();
                $asset = 'com_uithemebuilderlite.category.' . $this->_item->id;

                // Check general create permission.
                if ($user->authorise('core.create', $asset)) {
                    $this->_item->getParams()->set('access-create', true);
                }

                $this->_children = $this->_item->getChildren();
                $this->_parent = false;

                if ($this->_item->getParent()) {
                    $this->_parent = $this->_item->getParent();
                }

                $this->_rightsibling = $this->_item->getSibling();
                $this->_leftsibling = $this->_item->getSibling(false);
            } else {
                $this->_children = false;
                $this->_parent = false;
            }
        }

        return $this->_item;
    }

    /**
     * Get the parent category.
     *
     * @return  \Joomla\CMS\Categories\CategoryNode|null  An array of categories or false if an error occurs.
     * @throws \Exception
     *
     * @since  1.0.0
     */
    public function getParent(): ?CategoryNode
    {
        if (!is_object($this->_item)) {
            $this->getCategory();
        }
        return $this->_parent;
    }


    /**
     * Get the sibling (adjacent) categories.
     *
     * @return  mixed  An array of categories or false if an error occurs.
     *
     * @throws \Exception
     * @since  1.0.0
     */
    public function &getLeftSibling(): mixed
    {
        if (!is_object($this->_item)) {
            $this->getCategory();
        }
        return $this->_leftsibling;
    }


    /**
     * Get the sibling (adjacent) categories.
     *
     * @return  mixed  An array of categories or false if an error occurs.
     *
     * @throws \Exception
     * @since  1.0.0
     */
    public function &getRightSibling(): mixed
    {
        if (!is_object($this->_item)) {
            $this->getCategory();
        }
        return $this->_rightsibling;
    }


    /**
     * Get the child categories.
     *
     * @return  mixed  An array of categories or false if an error occurs.
     *
     * @throws \Exception
     *
     * @since  1.0.0
     */
    public function &getChildren(): mixed
    {
        if (!is_object($this->_item)) {
            $this->getCategory();
        }

        // Order subcategories
        if ($this->_children) {
            $params = $this->getState()->get('params');

            $orderByPri = $params->get('orderby_pri');

            if ($orderByPri === 'alpha' || $orderByPri === 'ralpha') {
                $this->_children = ArrayHelper::sortObjects($this->_children, 'title', ($orderByPri === 'alpha') ? 1 : (-1));
            }
        }

        return $this->_children;
    }


    /**
     * Generate column expression for slug or catslug.
     *
     * @param   \JDatabaseQuery  $query  Current query instance.
     * @param   string           $id     Column id name.
     * @param   string           $alias  Column alias name.
     *
     * @return  string
     *
     * @since   v1.0.0
     */
    private function getSlugColumn(DatabaseQuery $query, string $id, string $alias): string
    {
        return 'CASE WHEN '
            . $query->charLength($alias, '!=', '0')
            . ' THEN '
            . $query->concatenate([$query->castAsChar($id), $alias], ':')
            . ' ELSE '
            . $query->castAsChar($id) . ' END';
    }


    /**
     * Increment the hit counter for the category.
     *
     * @param   integer  $pk  Optional primary key of the category to increment.
     *
     * @return  boolean  True if successful; false otherwise and internal error set.
     *
     * @throws \Exception
     * @since   v1.0.0
     */
    public function hit($pk = 0)
    {
        $input = Factory::getApplication()->input;
        $hitCount = $input->getInt('hitcount', 1);

        if ($hitCount) {
            $pk = (!empty($pk)) ? $pk : (int) $this->getState('category.id');

            $table = Table::getInstance('Category', 'JTable');
            $table->hit($pk);
        }

        return true;
    }
}
