<?php
/**
 * @package      ui/Theme-Builder Lite
 * @author       Stephan W.
 * @author       url   https://www.ui-themebuilder.com/
 * @copyright    Copyright (C) 2021 ui-themebuilder.com, All rights reserved.
 * @developer    Stephan Wittling - https://www.ui-themebuilder.com/
 *               ui/Theme-Builder Lite is distributed in the hope that it will be useful,
 *               but WITHOUT ANY WARRANTY; without even the implied warranty of
 *               MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 *               See the GNU General Public License for more details.
 * @license      http://www.gnu.org/licenses/gpl.html GNU/GPL
 *********************************************************************************/

namespace SW\Component\uiThemeBuilderLite\Site\Helper;

defined('_JEXEC') or die;


use Joomla\CMS\Component\ComponentHelper;
use Joomla\CMS\Factory;
use Joomla\CMS\Filesystem\File;
use Joomla\Filesystem\Folder;
use Joomla\CMS\Helper\ContentHelper;
use Joomla\CMS\Router\Route;
use Joomla\CMS\Uri\Uri;

/**
 * ui/Theme-Builder Lite component site helper.
 *
 * @since  v1.0.0
 */
class SitePageHelper extends ContentHelper
{

    protected static array $module = [];

    /**
     * Get Component classes.
     *
     * @param   array  $options
     *
     * @return string
     *
     * @since  v1.2.0
     */
    public static function getClasses(array $options): string
    {
        $classes = array_filter([
            $options['show'] ?? '',
            $options['hidden'] ?? '',
            $options['class'] ?? '',
            $options['modul_padding'] ?? '',
            $options['modul_shadow'] ?? '',
            $options['alignment'] ?? '',
            $options['modul_height'] ?? '',
            $options['flex_content_vertical'] ?? '',
            $options['color_style'] ?? '',
            (($options['use_img_transition'] ?? 0) === '1' || ($options['use_overlay_transition'] ?? 0) === '1') ? 'uk-transition-toggle' : ''
        ]);

        return implode(' ', $classes);
    }

    /**
     * Get Title and/or Subtitle classes.
     *
     * @param   array  $options
     *
     * @return array
     *
     * @since  v1.2.0
     */
    public static function getTitleAndSubtitleClasses(array $options): array
    {

        if (!is_array($options)) {
            $options = [];
        }
        $classes = ['title' => '', 'subtitle' => ''];

        if (!empty($options['title'])) {
            $title_classes    = array_filter([
                $options['title_font_size'] ?? '',
                $options['title_font_weight'] ?? '',
                $options['title_style'] ?? '',
                $options['title_text_color'] ?? '',
                $options['title_text_transform'] ?? '',
                ($options['title_font_family_handwriting'] ?? '') === "1" ? 'tm-handwriting-font' : '',
                !empty($options['title_width_viewport']) && $options['image_height'] === 'viewport' && ($options['image_typ'] ?? 1) === "1" ? 'uk-width-' . $options['title_width_viewport'] : ''
            ]);
            $classes['title'] = ' ' . implode(' ', $title_classes);
        }

        if (!empty($options['subtitle'])) {
            $subtitle_classes    = array_filter([
                $options['subtitle_font_size'] ?? '',
                $options['subtitle_font_weight'] ?? '',
                $options['subtitle_text_color'] ?? '',
                $options['subtitle_title_style'] ?? '',
                $options['subtitle_text_transform'] ?? '',
            ]);
            $classes['subtitle'] = ' ' . implode(' ', $subtitle_classes);
        }

        return $classes;
    }

    /**
     * Output Components title
     *
     * @param   string  $titleTag
     * @param   string  $class
     * @param   string  $titleName
     * @param   string  $titleClassName
     *
     * @return bool|string
     *
     * @throws \DOMException
     *
     * @since  v1.4.4
     */
    public static function setTitle(string $titleTag = 'h3', string $class = '', string $titleName = '', string $titleClassName = 'ui-title'): bool|string
    {
        $dom = new \DOMDocument('1.0', 'UTF-8');
        $title = $dom->createElement($titleTag);
        $title->setAttribute('class', ((trim($class) === '') ? $titleClassName : ($class . ' ' . $titleClassName)));

        $span = $dom->createElement('span', $titleName); // The 2nd argument populates the textContent
        $title->appendChild($span);

        $dom->appendChild($title);

        return $dom->saveHTML($title);
    }

    /**
     * Get a random unsplash image url.
     *
     * @param   array   $randomImageSizes
     * @param   string  $randomImageKeywords
     *
     * @return string
     *
     * @since  v1.2.0
     */
    public static function getRandomUnsplashUrl(array $randomImageSizes = ['800x600', '1024x768', '1920x1080'], string $randomImageKeywords = 'nature,water'): string
    {
        $randomSize = $randomImageSizes[array_rand($randomImageSizes)];

        return 'https://source.unsplash.com/random/' . $randomSize . '/?' . trim($randomImageKeywords);
    }

    /**
     * @param   null  $src
     * @param   int   $width
     * @param   null  $height
     * @param   int   $quality
     * @param   bool  $crop
     * @param   bool  $cacheDir
     * @param   bool  $directOutput
     *
     * @return bool|string
     *
     * @since  v1.0.0
     */
    public static function image_resize($src = null, int $width = 100, $height = null, int $quality = 80, bool $crop = false, bool|string $cacheDir = false, bool $directOutput = true): bool|string
    {
        $img = (!File::exists($src) ? getimagesize($src) : '');
        if ($img && extension_loaded('gd')) {
            if (!list($w, $h) = getimagesize($src)) {
                return "Unsupported picture type!";
            }
            $type = strtolower(substr(strrchr($src, "."), 1));
            if ($type === 'jpeg') {
                $type = 'jpg';
            }
            $file_name = 'th_' . pathinfo($src, PATHINFO_FILENAME) . '_' . $width;
            if ($cacheDir) {
                if (!file_exists($cacheDir)) {
                    Folder::create($cacheDir);
                    die();
                }
                // Check if cached file exists:
                $dst = $cacheDir . '/' . $file_name . '.' . $type;
                if (is_file($dst)) {
                    if ($directOutput) {
                        switch ($type) {
                            case 'bmp':
                                $img = imagecreatefrombmp($dst);
                                imagewbmp($img);
                                break;
                            case 'gif':
                                $img = imagecreatefromgif($dst);
                                imagegif($img);
                                break;
                            case 'png':
                                $img = imagecreatefrompng($dst);
                                imagepng($img);
                                break;
                            case 'jpg':
                                $img = imagecreatefromjpeg($dst);
                                imagejpeg($img, null, $quality);
                                break;
                        }
                        die();
                    }

                    return $dst;
                }
            }

            switch ($type) {
                case 'bmp':
                    $img = imagecreatefrombmp($src);
                    break;
                case 'gif':
                    $img = imagecreatefromgif($src);
                    break;
                case 'png':
                    $img = imagecreatefrompng($src);
                    break;
                case 'jpg':
                    $img = imagecreatefromjpeg($src);
                    break;
                default:
                    return "Unsupported picture type!";
            }

            // Calculate missing dimensions:
            if ($width === null || $width === 0) {
                $width = $w * ($height / $h);
            } elseif ($height === null || $height === 0) {
                $height = $h * ($width / $w);
            }

            // resize
            if ($crop) {
                $ratio = max($width / $w, $height / $h);
                $x     = 0;
                $y     = 0;
                if ($width / $height > $w / $h) {
                    $y = ($h - ($height / $ratio)) / 2;
                } else {
                    $x = ($w - ($width / $ratio)) / 2;
                }
                $h = $height / $ratio;
                $w = $width / $ratio;
            } else {
                $ratio  = min($width / $w, $height / $h);
                $width  = $w * $ratio;
                $height = $h * $ratio;
                $x      = 0;
                $y      = 0;
            }

            $new = imagecreatetruecolor($width, $height);

            // preserve transparency
            if ($type === "gif" || $type === "png") {
                imagecolortransparent($new, imagecolorallocatealpha($new, 0, 0, 0, 127));
                imagealphablending($new, false);
                imagesavealpha($new, true);
            }

            self::newimagecopyresampled($new, $img, 0, 0, $x, $y, $width, $height, $w, $h);

            if ($cacheDir) {
                // Save file
                $dst = $cacheDir . '/' . $file_name . '.' . $type;
                switch ($type) {
                    case 'bmp':
                        imagewbmp($new, $dst);
                        break;
                    case 'gif':
                        imagegif($new, $dst);
                        break;
                    case 'jpg':
                        imagejpeg($new, $dst, $quality);
                        break;
                    case 'png':
                        imagepng($new, $dst);
                        break;
                }
            }

            if ($directOutput) {
                switch ($type) {
                    case 'bmp':
                        header('Content-Type: image/bmp');
                        imagewbmp($new);
                        break;
                    case 'gif':
                        header('Content-Type: image/gif');
                        imagegif($new);
                        break;
                    case 'jpg':
                        header('Content-Type: image/jpeg');
                        imagejpeg($new, null, $quality);
                        break;
                    case 'png':
                        header('Content-Type: image/png');
                        imagepng($new);
                        break;
                }

                return true;
            }

            return $dst ?? false;
        }

        return "Wrong path! or The GD library is not installed.";
    }

    /**
     * @param        $dst_image
     * @param        $src_image
     * @param        $dst_x
     * @param        $dst_y
     * @param        $src_x
     * @param        $src_y
     * @param        $dst_w
     * @param        $dst_h
     * @param        $src_w
     * @param        $src_h
     * @param   int  $quality
     *
     * @return bool
     *
     * @since  v1.0.0
     */
    public static function newimagecopyresampled($dst_image, $src_image, $dst_x, $dst_y, $src_x, $src_y, $dst_w, $dst_h, $src_w, $src_h, int $quality = 3): bool
    {
        if ((empty($src_image) || empty($dst_image)) && !extension_loaded('gd')) {
            return false;
        }
        if ($quality < 5 &&
            ($dst_w * $quality < $src_w || $dst_h * $quality < $src_h)
        ) {
            $tmp_w = $dst_w * $quality;
            $tmp_h = $dst_h * $quality;
            $temp  = imagecreatetruecolor($tmp_w + 1, $tmp_h + 1);
            imagecopyresized(
                $temp,
                $src_image,
                $dst_x * $quality,
                $dst_y * $quality,
                $src_x,
                $src_y,
                $tmp_w + 1,
                $tmp_h + 1,
                $src_w,
                $src_h
            );
            imagecopyresampled(
                $dst_image,
                $temp,
                0,
                0,
                0,
                0,
                $dst_w,
                $dst_h,
                $tmp_w,
                $tmp_h
            );
            imagedestroy($temp);
        } else {
            imagecopyresampled(
                $dst_image,
                $src_image,
                (int) $dst_x,
                (int) $dst_y,
                (int) $src_x,
                (int) $src_y,
                (int) $dst_w,
                (int) $dst_h,
                (int) $src_w,
                (int) $src_h
            );
        }

        return true;
    }

    /**
     * @param          $content
     * @param   array  $targetClasses
     *
     * @return string
     * @throws \JsonException
     * @since  v1.6.4
     */
    public static function addImageMicrodata($content, array $targetClasses = []): string
    {

        if (empty($targetClasses) || empty($content)) {
            return $content;
        }

        // Create a DOMDocument from the HTML content
        $dom = new \DOMDocument;
        libxml_use_internal_errors(true);
        $dom->loadHTML($content);
        libxml_use_internal_errors(false);

        $xpath = new \DOMXPath($dom);

        $images             = [];
        $processedFilenames = [];

        // Iterate through the target classes
        foreach ($targetClasses as $class) {
            $elements = $xpath->query("//*[contains(concat(' ', normalize-space(@class), ' '), ' $class ')]");

            foreach ($elements as $element) {
                $imgTags = $element->getElementsByTagName('img');

                if ($imgTags->length > 0) {
                    foreach ($imgTags as $imgTag) {
                        $alt = $imgTag->getAttribute('alt');
                        $altParts = explode('|', $alt);

                        if (count($altParts) >= 2) {
                            $description = trim($altParts[0]);
                            $creatorName = trim($altParts[1]);

                            $license = isset($altParts[2]) ? htmlspecialchars(trim($altParts[2]), ENT_QUOTES) : '';
                            $acquireLicensePage = isset($altParts[3]) ? htmlspecialchars(trim($altParts[3]), ENT_QUOTES) : '';
                            $creditText = isset($altParts[4]) ? htmlspecialchars(trim($altParts[4]), ENT_QUOTES) : '';

                            $contentUrl = $imgTag->getAttribute('data-src') ?: $imgTag->getAttribute('src');

                            if ($contentUrl) {
                                $filename = basename($contentUrl);

                                // If the file name is already included, no microdata will be generated.
                                if (!in_array($filename, $processedFilenames, true)) {
                                    $jsonLd = [
                                        "@context" => "https://schema.org/",
                                        "@type" => "ImageObject",
                                        "contentUrl" => htmlspecialchars($contentUrl, ENT_QUOTES),
                                        "creator" => [
                                            "@type" => "Person",
                                            "name" => $creatorName,
                                        ],
                                    ];

                                    if (!empty($license)) {
                                        $jsonLd["license"] = $license;
                                    }

                                    if (!empty($acquireLicensePage)) {
                                        $jsonLd["acquireLicensePage"] = $acquireLicensePage;
                                    }

                                    if (!empty($creditText)) {
                                        $jsonLd["creditText"] = $creditText;
                                    }

                                    $images[] = $jsonLd;
                                    $processedFilenames[] = $filename;
                                }
                            }
                            $imgTag->setAttribute('alt', $description);
                        }
                    }
                }
            }
        }

        if (!empty($images)) {
            // Create JSON-LD microdata for all images
            $jsonScript = '<script type="application/ld+json">' . json_encode($images, JSON_THROW_ON_ERROR | JSON_UNESCAPED_SLASHES | JSON_UNESCAPED_UNICODE) . '</script>';

            // Add JSON-LD microdata as a script before the first <img> tag found.
            $content = preg_replace('/<img[^>]+>/', $jsonScript . '$0', $content, 1);
        }

        return $content;
    }

    /**
     * @param           $components
     * @param   string  $modID
     * @param   bool    $cache
     * @param   string  $cacheTime
     * @param   string  $modGridChildWidth
     * @param   string  $modGrid
     *
     * @return string
     *
     * @throws \Exception
     *
     * @since  v1.0.0
     */
    public static function loadModComponents($components, string $modID, bool $cache, string $cacheTime, string $modGridChildWidth = '1-1', string $modGrid = 'small'): string
    {
        if (!$components) {
            return true;
        }

        $html = '<div class="uk-grid-' . $modGrid . ' uk-child-width-' . $modGridChildWidth . '@m" uk-grid="">';
        $html .= ComponentParser::viewComponents($components, $modID, $cache, $cacheTime);
        $html .= '</div>';

        // optimize HTML Output
        return trim(preg_replace('/^\s+|\n|\r|\s+$/m', '', $html));
    }

    /**
     * @param $obj
     *
     * @return array|mixed
     *
     * @since v1.0.0
     */
    public static function toArray($obj): mixed
    {
        if (is_object($obj)) {
            $obj = (array) $obj;
        }
        if (is_array($obj)) {
            $new_array = array ();
            foreach ($obj as $key => $val) {
                $new_array[$key] = self::toArray($val);
            }
        } else {
            $new_array = $obj;
        }

        return $new_array;
    }

    /**
     * @param $array
     *
     * @return mixed
     *
     * @since v1.0.0
     */
    public static function escapeHtmlArray($array): mixed
    {
        $not_escapable = array (
            'title',
            'button_text',
            'content',
            'item_content',
            'item_description_content',
            'item_text',
            'video_content'
        );
        foreach ($array as $key => $val) {
            if (!in_array($key, $not_escapable, true)) {
                $array[$key] = self::escapeHtml($val);
            }
        }

        return $array;
    }

    /**
     * @param $string
     *
     * @return array|string|null
     *
     * @since v1.0.0
     */
    public static function escapeHtml($string): array|string|null
    {
        $string = htmlentities($string, ENT_QUOTES, 'UTF-8');

        return preg_replace(
            '/&lt;span class=&quot;search-results&quot;&gt;(.*?)&lt;\/span&gt;/',
            '<span class="search-results">$1</span>',
            $string
        );
    }

    /**
     * @param                    $svg_path
     * @param   int|false        $key
     * @param   string|bool      $color_placeholder_1
     * @param   string|bool      $color_placeholder_2
     * @param   string|bool      $color_placeholder_3
     * @param   string|bool      $text_placeholder
     * @param   string|bool      $text_font_size_placeholder
     * @param   int|bool|string  $opacity_placeholder
     *
     * @return bool|string
     *
     * @since v1.0.0
     */
    public static function convertSVGtoURI($svg_path, int|bool $key = false, $color_placeholder_1 = false, $color_placeholder_2 = false, $color_placeholder_3 = false, bool|string $text_placeholder = false, bool|string $text_font_size_placeholder = false, int|bool|string $opacity_placeholder = false): bool|string
    {
        if (!$svg_path) {
            return true;
        }

        $params            = ComponentHelper::getParams('com_uithemebuilderlite');
        $global_page_cache = $params->get('global_page_cache');
        $global_cache_time = $params->get('global_cache_time');

        $cachePath = JPATH_ROOT . DIRECTORY_SEPARATOR . 'cache' . DIRECTORY_SEPARATOR . 'com_uithemebuilderlite' . DIRECTORY_SEPARATOR;  // use site cache folder, not administrator cache folder
        $cacheFile = $cachePath . md5(basename($svg_path, ".svg")) . '.php';

        // Load SVG and optimize for ui
        $data = file_get_contents($svg_path);

        if ($global_cache_time !== '0') {
            // Time for cache files: Component settings
            $time = $global_cache_time;
        } else {
            // Time for cache files: 60 × 60 × 24 = 24 Hours
            $time = 60 * 60 * 24;
        }

        if ($global_page_cache == 1 && !file_exists($cachePath)) {
            Folder::create($cachePath, 0755);
        }
        if ($global_page_cache == 1 && file_exists($cacheFile) && filemtime($cacheFile) > time() - $time) {
            $cacheContent = file_get_contents($cacheFile);
            $data         = str_replace('<?php die("Access Denied"); ?>', '', $cacheContent);
        } else {
            // Placeholder to change color / text / font size / opacity
            if ($color_placeholder_1) {
                $data = str_replace("[COLOR_1]", $color_placeholder_1, $data);
            }
            if ($color_placeholder_2) {
                $data = str_replace("[COLOR_2]", $color_placeholder_2, $data);
            }
            if ($color_placeholder_3) {
                $data = str_replace("[COLOR_3]", $color_placeholder_3, $data);
            }
            if ($text_placeholder) {
                $data = str_replace("[TEXT]", $text_placeholder, $data);
            }
            if ($text_font_size_placeholder) {
                $data = str_replace("[FONT_SIZE]", $text_font_size_placeholder, $data);
            }
            if ($text_font_size_placeholder) {
                $data = str_replace("[OPACITY_1]", $opacity_placeholder, $data);
            }

            // optimize SVG
            $data = preg_replace("/[\n\r]/", "", $data);
            $data = str_replace(array ('"', "%", "<", ">", " ", "!", "*", "(", ")", ";", ":", "@", "&", "=", "+", "$", ",", "/", "?", "#", "[", "]", '{', '}'), array ("'", "%25", "%3C", "%3E", "%20", "%21", "%2A", "%28", "%29", "%3B", "%3A", "%40", "%26", "%3D", "%2B", "%24", "%2C", "%2F", "%3F", "%23", "%5B", "%5D", '%7B', '%7D'), $data);
            $data = trim($data);

            // generated cache files
            if ($global_page_cache == 1) {
                file_put_contents($cacheFile, '<?php die("Access Denied"); ?>' . $data, LOCK_EX);

                // write index.html in cache folder if folder exists
                self::createIndexFile($cachePath);
            } elseif ($global_page_cache == 0 && file_exists($cacheFile)) {
                chmod($cacheFile, 0777);
                unlink($cacheFile);
            }
        }

        return $data;
    }

    /**
     * @param $array
     *
     * @return string
     *
     * @since v1.0.0
     */
    public static function buildStyle($array): string
    {
        $style = '';
        if (count($array)) {
            $style .= ' style="';
            foreach ($array as $key => $val) {
                if ($key === 'background-image' ||
                    $key === 'mask' ||
                    $key === '-webkit-mask'
                ) {
                    $style .=
                        $key .
                        ': url(' .
                        htmlspecialchars('\'' . $val . '\'', ENT_QUOTES) .
                        ');';
                } else {
                    $style .= $key . ':' . $val . ';';
                }
            }
            $style .= '"';
        }

        return $style;
    }

    /**
     *
     * @return int
     *
     * @throws \Exception
     * @since v1.0.0
     */
    public static function randomNumber(): int
    {
        return random_int(1000, 9999);
    }

    /**
     * @param   int         $count
     * @param   string      $article_ordering
     * @param   int|string  $catid
     * @param   bool        $include_subcategories
     * @param   int|string  $tagid
     *
     * @return array|mixed
     *
     * @throws \Exception
     *
     * @since v1.0.0
     */
    public static function getArticles(int $count = 2, string $article_ordering = 'latest', int|string $catid = '', bool $include_subcategories = true, int|string $tagid = ''): mixed
    {
        $authorised = \JAccess::getAuthorisedViewLevels(
            Factory::getApplication()->getIdentity()->get('id')
        );
        $app        = Factory::getApplication();
        $db         = Factory::getContainer()->get('DatabaseDriver');
        $nullDate   = $db->quote($db->getNullDate());
        $nowDate    = $db->quote(Factory::getDate()->toSql());
        $query      = $db->getQuery(true);
        $query
            ->select('a.*')
            ->from($db->quoteName('#__content', 'a'))
            ->select($db->quoteName('b.alias', 'category_alias'))
            ->select($db->quoteName('b.title', 'category'))
            ->join(
                'LEFT',
                $db->quoteName('#__categories', 'b') .
                ' ON (' .
                $db->quoteName('a.catid') .
                ' = ' .
                $db->quoteName('b.id') .
                ')'
            )
            ->where(
                $db->quoteName('b.extension') . ' = ' . $db->quote('com_content')
            );
        $query->where($db->quoteName('a.state') . ' = ' . $db->quote(1));
        if ($catid !== '' || is_array($catid)) {
            if (!is_array($catid)) {
                $catid = array ($catid);
            }
            if (!in_array('', $catid, true)) {
                $categories = self::getJcategories($catid, $include_subcategories);
                $categories = array_merge($categories, $catid);
                $query->where(
                    $db->quoteName('a.catid') . " IN (" . implode(',', $categories) . ")"
                );
            }
        }
        // tags filter
        if ($tagid) {
            $subQuery = $db
                ->getQuery(true)
                ->select('DISTINCT content_item_id')
                ->from($db->quoteName('#__contentitem_tag_map'))
                ->where('tag_id IN (' . $tagid . ')')
                ->where('type_alias = ' . $db->quote('com_content.article'));
            $query->innerJoin(
                '(' .
                (string) $subQuery .
                ') AS tagmap ON tagmap.content_item_id = a.id'
            );
        }

        $query->where(
            '(a.publish_up = ' . $nullDate . ' OR a.publish_up <= ' . $nowDate . ')'
        );
        // $query->where('(a.publish_down = ' . $nullDate . ' OR a.publish_down >= ' . $nowDate . ')');
        if ($article_ordering === 'hits') {
            $query->order($db->quoteName('a.hits') . ' DESC');
        } elseif ($article_ordering === 'featured') {
            $query->where($db->quoteName('a.featured') . ' = ' . $db->quote(1));
            $query->order($db->quoteName('a.publish_up') . ' DESC');
        } elseif ($article_ordering === 'oldest') {
            $query->order($db->quoteName('a.publish_up') . ' ASC');
        } elseif ($article_ordering === 'alphabet_asc') {
            $query->order($db->quoteName('a.title') . ' ASC');
        } elseif ($article_ordering === 'alphabet_desc') {
            $query->order($db->quoteName('a.title') . ' DESC');
        } elseif ($article_ordering === 'random') {
            $query->order($query->Rand());
        } else {
            $query->order($db->quoteName('a.publish_up') . ' DESC');
        }
        if ($app->getLanguageFilter()) {
            $query->where(
                'a.language IN (' .
                $db->Quote($app->getLanguage()->getTag()) .
                ',' .
                $db->Quote('*') .
                ')'
            );
        }
        // $query->where($db->quoteName('a.access') . " IN (" . implode(',', $authorised) . ")");
        $query->order($db->quoteName('a.created') . ' DESC')->setLimit($count);
        $db->setQuery($query);

        $items = $db->loadObjectList();
        foreach ($items as &$item) {
            $item->slug          = $item->id . ':' . $item->alias;
            $item->catslug       = $item->catid . ':' . $item->category_alias;
            $item->username      = Factory::getApplication()->getIdentity()->username;
            $item->category_link = urldecode(
                Route::_(\ContentHelperRoute::getCategoryRoute($item->catslug))
            );

            if (in_array($item->access, $authorised, true)) {
                // user has the privilege to view the article
                $item->link = Route::_(
                    \ContentHelperRoute::getArticleRoute(
                        $item->slug,
                        $item->catid,
                        $item->language
                    )
                );
            } else {
                $item->link = new Uri(
                    Route::_('index.php?option=com_users&view=login', false)
                );
                $item->link->setVar(
                    'return',
                    base64_encode(
                        \ContentHelperRoute::getArticleRoute(
                            $item->slug,
                            $item->catid,
                            $item->language
                        )
                    )
                );
                // $item->linkText = \JText::_('MOD_ARTICLES_NEWS_READMORE_REGISTER');
            }

            $itemImages            = trim($item->images);
            $images                = json_decode($itemImages, false, 512, JSON_THROW_ON_ERROR);
            $item->image_thumbnail = '';
            $item->imageAlt        = '';
            if (!empty($images->image_intro)) {
                $item->image_thumbnail =
                    URI::root() .
                    htmlspecialchars($images->image_intro, ENT_COMPAT, 'UTF-8');
                $item->imageAlt        = htmlspecialchars(
                    $images->image_intro_alt,
                    ENT_COMPAT,
                    'UTF-8'
                );
            }
        }

        return $items;
    }

    /**
     * @param   int    $parent_id
     * @param   bool   $include_subcategories
     * @param   bool   $child
     * @param   array  $cats
     *
     * @return array|mixed
     *
     * @throws \Exception
     * @since v0.0.1
     */
    public static function getJcategories($parent_id = 1, bool $include_subcategories = true, bool $child = false, array $cats = array ()): mixed
    {
        $db    = Factory::getContainer()->get('DatabaseDriver');
        $query = $db->getQuery(true);
        $query
            ->select('*')
            ->from($db->quoteName('#__categories'))
            ->where($db->quoteName('extension') . ' = ' . $db->quote('com_content'))
            ->where($db->quoteName('published') . ' = ' . $db->quote(1))
            ->where(
                $db->quoteName('access') .
                " IN (" .
                implode(',', Factory::getApplication()->getIdentity()->getAuthorisedViewLevels()) .
                ")"
            )
            ->where(
                $db->quoteName('language') .
                " IN (" .
                $db->Quote(Factory::getApplication()->getLanguage()->getTag()) .
                ", " .
                $db->Quote('*') .
                ")"
            )
            ->where(
                $db->quoteName('parent_id') . " IN (" . implode(',', $parent_id) . ")"
            )
            ->order($db->quoteName('lft') . ' ASC');
        $db->setQuery($query);
        $rows = $db->loadObjectList();
        foreach ($rows as $row) {
            if ($include_subcategories) {
                $cats[] = $row->id;
                if (self::hasJchildren($row->id)) {
                    $cats = self::getJcategories(
                        array ($row->id),
                        $include_subcategories,
                        true,
                        $cats
                    );
                }
            }
        }

        return $cats;
    }

    /**
     * @param   $parent_id
     *
     * @return bool
     *
     * @throws \Exception
     *
     * @since v1.0.0
     */
    private static function hasJchildren($parent_id = 1): bool
    {
        $db    = Factory::getContainer()->get('DatabaseDriver');
        $query = $db->getQuery(true);
        $query
            ->select('*')
            ->from($db->quoteName('#__categories'))
            ->where($db->quoteName('extension') . ' = ' . $db->quote('com_content'))
            ->where($db->quoteName('published') . ' = ' . $db->quote(1))
            ->where(
                $db->quoteName('access') .
                " IN (" .
                implode(',', Factory::getApplication()->getIdentity()->getAuthorisedViewLevels()) .
                ")"
            )
            ->where(
                $db->quoteName('language') .
                " IN (" .
                $db->Quote(Factory::getApplication()->getLanguage()->getTag()) .
                ", " .
                $db->Quote('*') .
                ")"
            )
            ->where($db->quoteName('parent_id') . ' = ' . $db->quote($parent_id))
            ->order($db->quoteName('created_time') . ' DESC');
        $db->setQuery($query);
        $childrens = $db->loadObjectList();

        return is_array($childrens) && count($childrens);
    }

    /**
     * @param        $id
     * @param   int  $count
     *
     * @return array|false
     *
     * @throws \JsonException
     *
     * @since v0.0.1
     */
    public static function getFlickrImages($id, int $count = 5): bool|array
    {
        $params            = ComponentHelper::getParams('com_uithemebuilderlite');
        $global_page_cache = $params->get('global_page_cache');
        $global_cache_time = $params->get('global_cache_time');

        $cachePath = JPATH_ROOT . DIRECTORY_SEPARATOR . 'cache' . DIRECTORY_SEPARATOR . 'com_uithemebuilderlite' . DIRECTORY_SEPARATOR;  // use site cache folder, not administrator cache folder
        $cacheFile = $cachePath . 'flickr.json';

        if ($global_page_cache == 1 && !file_exists($cachePath)) {
            Folder::create($cachePath, 0755);
        }

        if ($global_cache_time !== '0') {
            // Time for cache files: Component settings
            $time = $global_cache_time;
        } else {
            // Time for cache files: 60 × 60 × 24 = 24 Hours
            $time = 60 * 60 * 24;
        }

        // use cache files
        if ($global_page_cache == 1 && file_exists($cacheFile) && filemtime($cacheFile) > time() - $time) {
            $images = file_get_contents($cacheFile);
        } else {
            $url = 'https://api.flickr.com/services/feeds/photos_public.gne?id=' . $id . '&format=json&nojsoncallback=1';
            if ($url) {
                if (ini_get('allow_url_fopen')) {
                    $images = file_get_contents($url);
                } else {
                    $images = (new self)->curl($url);
                }
                if ($global_page_cache == 1) {
                    file_put_contents($cacheFile, $images, LOCK_EX);

                    // write index.html in cache folder if folder exists
                    self::createIndexFile($cachePath);
                } elseif ($global_page_cache == 0 && file_exists($cacheFile)) {
                    chmod($cacheFile, 0777);
                    unlink($cacheFile);
                }
            } else {
                return false;
            }
        }

        $itemImages = trim($images);
        $json       = json_decode($itemImages, false, 512, JSON_THROW_ON_ERROR);

        return $json->items ?? array ();
    }

    /**
     * @param $url
     *
     * @return bool|string
     *
     * @since v0.0.1
     */
    function curl($url): bool|string
    {
        $x = curl_init();
        curl_setopt($x, CURLOPT_URL, $url);
        curl_setopt($x, CURLOPT_RETURNTRANSFER, 1);
        $data = curl_exec($x);
        curl_close($x);

        return $data;
    }

    /**
     * @param $directory
     *
     * @return bool
     *
     * @since v1.6.2
     */
    public static function createIndexFile($directory): bool
    {
        $indexContent   = '<!DOCTYPE html><title></title>';
        $indexPath      = $directory . DIRECTORY_SEPARATOR . 'index.html';

        if (!file_exists($indexPath) && is_dir($directory)) {
            file_put_contents($indexPath, $indexContent, LOCK_EX);
            chmod($indexPath, 0775);
            return true;
        } else {
            return false;
        }
    }
}
