<?php
/**
 * @package      ui/Theme-Builder Lite
 * @author       Stephan W.
 * @author       url   https://www.ui-themebuilder.com/
 * @copyright    Copyright (C) 2021 ui-themebuilder.com, All rights reserved.
 * @developer    Stephan Wittling - https://www.ui-themebuilder.com/
 *               ui/Theme-Builder Lite is distributed in the hope that it will be useful,
 *               but WITHOUT ANY WARRANTY; without even the implied warranty of
 *               MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 *               See the GNU General Public License for more details.
 * @license      http://www.gnu.org/licenses/gpl.html GNU/GPL
 *********************************************************************************/

namespace SW\Component\uiThemeBuilderLite\Site\View\Page;

defined('_JEXEC') or die;

use Joomla\CMS\Component\ComponentHelper;
use Joomla\CMS\Factory;
use Joomla\CMS\Filesystem\File;
use Joomla\CMS\HTML\HTMLHelper;
use Joomla\CMS\Language\Text;
use Joomla\CMS\MVC\View\GenericDataException;
use Joomla\CMS\MVC\View\HtmlView as BaseHtmlView;
use Joomla\CMS\Uri\Uri;
use SW\Component\uiThemeBuilderLite\Administrator\Extension\uiThemeBuilderLiteComponent;

/**
 * HTML Page View class for the ui/Theme-Builder Lite component
 *
 * @since  v1.0.0
 */
class HtmlView extends BaseHtmlView
{

    protected $item;

    /**
     * @var mixed
     *
     * @since v1.0.0
     */
    public mixed $page;

    /**
     * Execute and display a template script.
     *
     * @param   string  $tpl  The name of the template file to parse; automatically searches through the template paths.
     *
     * @return  void  A string if successful, otherwise an Error object.
     *
     * @throws \Exception
     *
     * @since  v1.0.0
     */
    public function display($tpl = null): void
    {

        $app = Factory::getApplication();
        $this->page = $this->get('Item');
        $state = $this->get('State');
        // $user = Factory::getApplication()->getIdentity();

        $currentDate = Factory::getDate()->format('Y-m-d H:i:s');
        $publish_down = $this->page->publish_down == '0000-00-00 00:00:00' ? null : $this->page->publish_down;
        $isExpired = !is_null($publish_down) && $this->page->publish_down < $currentDate;
        $isNotPublishedYet = $this->page->publish_up > $currentDate;
        $isUnpublished = $this->page->published == uiThemeBuilderLiteComponent::CONDITION_UNPUBLISHED || $isNotPublishedYet || $isExpired;


        // Load UiKit files for non uikit templates without uithemebuilder Template
        $uikit_template_css = JPATH_ROOT . '/templates/uithemebuilder/css/theme.css';
        $uikit_css = JPATH_ROOT . '/media/com_uithemebuilderlite/css/uikit.min.css';
        if (File::exists($uikit_css) && !File::exists($uikit_template_css)) {
            HTMLHelper::_('stylesheet', 'com_uithemebuilderlite/uikit.min.css', array('version' => 'auto', 'relative' => true));
        }

        $uikit_template_js = JPATH_ROOT . '/templates/uithemebuilder/js/uikit.min.js';
        $uikit = JPATH_ROOT . '/media/com_uithemebuilderlite/js/uikit.min.js';
        if (File::exists($uikit) && !File::exists($uikit_template_js)) {
            HTMLHelper::_('script', 'com_uithemebuilderlite/uikit.min.js', array('version' => 'auto', 'relative' => true));
        }

        $uikit_icons_template_js = JPATH_ROOT . '/templates/uithemebuilder/js/uikit-icons.min.js';
        $uikit_icons = JPATH_ROOT . '/media/com_uithemebuilderlite/js/uikit-icons.min.js';
        if (File::exists($uikit_icons) && !File::exists($uikit_icons_template_js)) {
            HTMLHelper::_('script', 'com_uithemebuilderlite/uikit-icons.min.js', array('version' => 'auto', 'relative' => true));
        }

        Text::script('COM_UITHEMEBUILDERLITE_WARNING');

        // Component translations
        Text::script('COM_UITHEMEBUILDERLITE_DAYS');
        Text::script('COM_UITHEMEBUILDERLITE_HOURS');
        Text::script('COM_UITHEMEBUILDERLITE_MINUTES');
        Text::script('COM_UITHEMEBUILDERLITE_SECONDS');
        Text::script('COM_UITHEMEBUILDERLITE_HITS');
        Text::script('COM_UITHEMEBUILDERLITE_NO_ARTICLES_FOUND');
        Text::script('COM_UITHEMEBUILDERLITE_POSTED_IN');
        Text::script('COM_UITHEMEBUILDERLITE_WRITTEN_BY');

        Text::script('COM_UITHEMEBUILDERLITE_LOAD_MORE');
        Text::script('COM_UITHEMEBUILDERLITE_LOAD_MORE_RESET');

        // Component Gallery Pro
        Text::script('COM_UITHEMEBUILDERLITE_NEXT_PAGE');
        Text::script('COM_UITHEMEBUILDERLITE_PREVIOUS_PAGE');

        // Component Ajax Contact Form Pro
        Text::script('COM_UITHEMEBUILDERLITE_LABEL_GENDER');
        Text::script('COM_UITHEMEBUILDERLITE_LABEL_MALE');
        Text::script('COM_UITHEMEBUILDERLITE_LABEL_FEMALE');
        Text::script('COM_UITHEMEBUILDERLITE_LABEL_NAME');
        Text::script('COM_UITHEMEBUILDERLITE_LABEL_EMAIL');
        Text::script('COM_UITHEMEBUILDERLITE_LABEL_TEL');
        Text::script('COM_UITHEMEBUILDERLITE_LABEL_SUBJECT');
        Text::script('COM_UITHEMEBUILDERLITE_LABEL_MESSAGE');
        Text::script('COM_UITHEMEBUILDERLITE_LABEL_MESSAGE_PLACEHOLDER');
        Text::script('COM_UITHEMEBUILDERLITE_LABEL_DEFAULT_CAPTCHA');

        // Clear search
        $app->setUserState('uithemebuilderlite.search', '');

        // Check the page dates.
        if ($isExpired) {
            $app->enqueueMessage(Text::_('JLIB_HTML_PUBLISHED_EXPIRED_ITEM'), 'notice');
            $app->setHeader('status', 403, true);

            return;
        } elseif ($isUnpublished) {
            $app->enqueueMessage(Text::_('JERROR_ALERTNOAUTHOR'), 'error');
            $app->setHeader('status', 403, true);

            return;
        }

        if (count($errors = $this->get('Errors'))) {
            throw new GenericDataException(implode("\n", $errors), 500);
        }

        // Get the page/component configuration
        // $params = &$state->params;

        $active = Factory::getApplication()->getMenu()->getActive();

        // Override the layout only if this is not the active menu item
        // If it is the active menu item, then the view and item id will match
        if ((!$active) || ((!str_contains($active->link, 'view=page')) || (!str_contains($active->link, '&id=' . (string) $this->page->id)))) {
            $item = '';
            if (($layout = $item->params->get('page_layout'))) {
                $this->setLayout($layout);
            }
        } elseif (isset($active->query['layout'])) {
            // We need to set the layout in case this is an alternative menu item (with an alternative layout)
            $this->setLayout($active->query['layout']);
        }

        Factory::getApplication()->triggerEvent('onContentPrepare', ['com_uithemebuilderlite.page', &$this->page, &$this->page->content]);


        $this->_prepareDocument();

        parent::display($tpl);
    }

    /**
     * Prepares the document.
     *
     * @return  void
     *
     * @throws \Exception
     *
     * @since  v0.0.1
     */
    protected function _prepareDocument(): void
    {
        $config_params      = ComponentHelper::getParams('com_uithemebuilderlite');
        $app                = Factory::getApplication();
        $config             = $app;
        $doc                = $app->getDocument();
        $wa                 = $doc->getWebAssetManager();
        $menus              = $app->getMenu();
        $menu               = $menus->getActive()->getParams();

        if (isset($this->page->css) && $this->page->css) {
            $wa->addInlineStyle(trim(str_replace(array(
                "\r\n",
                "\r",
                "\n",
                '  '
            ), '', $this->page->css)));
        }

        $disable_og_url = $this->page->disable_og_url;

        // Page Title
        $title = ($menu ? $menu->get('page_title', '') : '');
        if (empty($title)) {
            $title = $this->page->title;
        }
        $page_title = $title;
        if ($config->get('sitename_pagetitles') == 2) {
            $page_title = $title . ' | ' . $config->get('sitename');
        } elseif ($config->get('sitename_pagetitles') == 1) {
            $page_title = $config->get('sitename') . ' | ' . $title;
        }

        $doc->setTitle($page_title);

        // Open Graph meta tags
        if ($this->page->open_graph_title) {
            $doc->setMetadata('og:title', $this->page->open_graph_title, 'property');
            $doc->setMetadata('og:site_name', $config->get('sitename'), 'property');
        }

        // Open Graph: Content type
        if ($this->page->open_graph_content_type) {
            $doc->setMetadata('og:type', $this->page->open_graph_content_type, 'property');
        }

        // Open Graph: Image
        if ($this->page->open_graph_image) {
            $doc->setMetadata('og:image', Uri::root() . $this->page->open_graph_image, 'property');
            [$width, $height] = getimagesize(Uri::root() . $this->page->open_graph_image);
            $doc->setMetadata('og:image:width', $width, 'property');
            $doc->setMetadata('og:image:height', $height, 'property');
            $info = getimagesize(Uri::root() . $this->page->open_graph_image);
            $type = $info['mime'];
            $doc->setMetadata('og:image:type', $type, 'property');
            if ($this->page->open_graph_image_alt) {
                $doc->setMetadata('og:image:alt', $this->page->open_graph_image_alt, 'property');
            }
        }

        // Open Graph: Description
        if ($this->page->open_graph_description) {
            $doc->setMetadata('og:description', $this->page->open_graph_description, 'property');
        }

        // Open Graph: url
        if (!$disable_og_url) {
            $doc->setMetadata('og:url', Uri::current(), 'property');
        }

        // Facebook api key
        $fb_app_id = $config_params->get('fb_app_id', '');
        if ($fb_app_id) {
            $doc->setMetadata('fb:app_id', $config_params->get('fb_app_id'), 'property');
        }

        // Twitter Card
        if ($this->page->twitter_card_content_type) {
            $doc->setMetadata('twitter:card', $this->page->twitter_card_content_type);
        }
        if ($this->page->twitter_card_site) {
            $doc->setMetadata('twitter:site', $this->page->twitter_card_site);
        }
        if ($this->page->twitter_card_description) {
            $doc->setMetadata('twitter:description', $this->page->twitter_card_description);
        }
        if ($this->page->twitter_card_image) {
            $doc->setMetadata('twitter:image:src', Uri::root() . $this->page->twitter_card_image);
            if ($this->page->twitter_card_image_alt) {
                $doc->setMetadata('twitter:image:alt', $this->page->twitter_card_image_alt);
            }
        }

        // custom page meta tags
        if ($this->page->metadesc) {
            $doc->setDescription($this->page->metadesc);
        } elseif ($menu && $menu->get('menu-meta_description')) {
            $doc->setDescription($menu->get('menu-meta_description'));
        }
        if ($this->page->metakey) {
            $doc->setMetadata('keywords', $this->page->metakey);
        } elseif ($menu && $menu->get('menu-meta_keywords')) {
            $doc->setMetadata('keywords', $menu->get('menu-meta_keywords'));
        }
        if ($this->page->robots) {
            $doc->setMetadata('robots', $this->page->robots);
        } elseif ($menu && $menu->get('robots')) {
            $doc->setMetadata('robots', $menu->get('robots'));
        }

        if ($app->get('MetaAuthor') === '1') {
            $author = $this->page->created_by;
            $user = Factory::getApplication()->getIdentity();
            $doc->setMetaData('author', $user);
        }
    }
}
