<?php
/**
 * @package      ui/Theme-Builder Lite
 * @author       Stephan W.
 * @author       url   https://www.ui-themebuilder.com/
 * @copyright    Copyright (C) 2021 ui-themebuilder.com, All rights reserved.
 * @developer    Stephan Wittling - https://www.ui-themebuilder.com/
 *               ui/Theme-Builder Lite is distributed in the hope that it will be useful,
 *               but WITHOUT ANY WARRANTY; without even the implied warranty of
 *               MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 *               See the GNU General Public License for more details.
 * @license      http://www.gnu.org/licenses/gpl.html GNU/GPL
 *********************************************************************************/

namespace SW\Component\uiThemeBuilderLite\Site\Model;

// No direct access
defined('_JEXEC') or die;

use Joomla\CMS\Factory;
use Joomla\CMS\Language\Multilanguage;
use Joomla\CMS\Language\Text;
use Joomla\CMS\MVC\Model\ItemModel;
use Joomla\CMS\Table\Table;
use Joomla\Database\ParameterType;

/**
 * Page model for the Joomla uiThemeBuilderLite component.
 *
 * @since  v1.0.0
 */
class PageModel extends ItemModel
{

    protected $_context = 'com_uithemebuilderlite.page';

    protected $_item = array();

    /**
     * Method to get page data.
     *
     * @param   null  $page_id
     *
     * @return  object|boolean  Menu item data object on success, boolean false
     *
     * @throws \Exception
     * @since   v1.0.0
     */
    public function getItem($page_id = null): object|bool
    {
        $page_id = (!empty($page_id)) ? $page_id : (int)$this->getState('page.id');

        if ($this->_item == null) {
            $this->_item = array();
        }

        if (!isset($this->_item[$page_id])) {
            try {
                $db = Factory::getContainer()->get('DatabaseDriver');
                $query = $db->getQuery(true)
                    ->select('a.*')
                    ->from('#__uithemebuilderlite as a')
                    ->where('a.id = ' . (int)$page_id);

                $query->select('l.title AS language_title')
                    ->leftJoin($db->quoteName('#__languages') . ' AS l ON l.lang_code = a.language');

                $query->select('ua.name AS author_name')
                    ->leftJoin('#__users AS ua ON ua.id = a.created_by');

                // Filter by published state.
                $published = $this->getState('filter.published');

                if (is_numeric($published)) {
                    $query->where('a.published = ' . (int)$published);
                } elseif ($published === '') {
                    $query->where('(a.published IN (0, 1))');
                }

                if ($this->getState('filter.language')) {
                    // $query->where('a.language in (' . $db->quote(Factory::getLanguage()->getTag()) . ',' . $db->quote('*') . ')');
                    $query->whereIn($db->quoteName('a.language'), [Factory::getApplication()->getLanguage()->getTag(), '*'], ParameterType::STRING);
                }

                $db->setQuery($query);
                $data = $db->loadObject();

                if (empty($data)) {
                    throw new \Exception(Text::_('JERROR_PAGE_NOT_FOUND'), 404);
                }

                // Convert parameter fields to objects.

                if ($access = $this->getState('filter.access')) {
                    $data->access_view = true;
                } else {
                    $user = Factory::getApplication()->getIdentity();
                    $groups = $user->getAuthorisedViewLevels();
                    $data->access_view = in_array($data->access, $groups, true);
                }

                $this->_item[$page_id] = $data;
            } catch (\Exception $e) {
                if ($e->getCode() == 404) {
                    // Need to go through the error handler to allow Redirect to work.
                    throw new \Exception($e->getMessage(), 404);
                } else {
                    $this->setError($e);
                    $this->_item[$pk] = false;
                }
            }
        }

        return $this->_item[$page_id];
    }

    /**
     * Increment the hit counter for the page.
     *
     * @param integer $pk Optional primary key of the page to increment.
     *
     * @return  boolean  True if successful; false otherwise and internal error set.
     *
     * @since   v1.0.0
     */
    public function hit($pk = 0): bool
    {

        $pk = (!empty($pk)) ? $pk : (int)$this->getState('page.id');
        $table = Table::getInstance('PageTable', 'SW\\Component\\uiThemeBuilderLite\\Administrator\\Table\\');
        $table->load($pk);
        $table->hit($pk);

        return true;
    }

    /**
     * Method to auto-populate the model state.
     *
     * Note. Calling getState in this method will result in recursion.
     *
     *
     * @return void
     *
     * @throws \Exception
     *
     * @since   v1.0.0
     */
    protected function populateState(): void
    {
        $app = Factory::getApplication();
        $page_id = $app->input->getInt('id');
        $this->setState('page.id', $page_id);

        $user = Factory::getApplication()->getIdentity();

        if (!$user->authorise('core.edit.state', 'com_uithemebuilderlite') && !$user->authorise('core.edit', 'com_uithemebuilderlite')) {
            $this->setState('filter.published', 1);
        }

        $this->setState('filter.language', Multilanguage::isEnabled());
    }
}
