<?php

/**
 * @package      ui/Theme-Builder Lite
 * @author       Stephan W.
 * @author       url   https://www.ui-themebuilder.com/
 * @copyright    Copyright (C) 2021-2023 ui-themebuilder.com, All rights reserved.
 * @developer    Stephan Wittling - https://www.ui-themebuilder.com/
 *               ui/Theme-Builder Lite is distributed in the hope that it will be useful,
 *               but WITHOUT ANY WARRANTY; without even the implied warranty of
 *               MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 *               See the GNU General Public License for more details.
 * @license      http://www.gnu.org/licenses/gpl.html GNU/GPL
 *********************************************************************************/

namespace SW\Component\uiThemeBuilderLite\Site\Helper;

// No direct access
defined('_JEXEC') or die;


use Joomla\CMS\Factory;
use Joomla\CMS\Filesystem\File;
use Joomla\Filesystem\Folder;
use Joomla\CMS\Layout\FileLayout;
use Joomla\CMS\Component\ComponentHelper;
use Joomla\CMS\Uri\Uri;
use SW\Component\uiThemeBuilderLite\Administrator\Helper\PageHelper;

class ComponentParser
{

    /**
     * Render Sections with columns
     *
     * @param                $content
     * @param   string|null  $id
     * @param   bool         $cache
     *
     * @return string|bool
     *
     * @throws \Exception
     * @since 0.0.1
     */
    public static function viewPage($content, ?string $id, bool $cache): string|bool
    {

        $params            = ComponentHelper::getParams('com_uithemebuilderlite');
        $global_page_cache = $params->get('global_page_cache');
        $global_cache_time = $params->get('global_cache_time');

        $cachePath = JPATH_ROOT . DIRECTORY_SEPARATOR . 'cache' . DIRECTORY_SEPARATOR . 'com_uithemebuilderlite' . DIRECTORY_SEPARATOR;  // use site cache folder, not administrator cache folder
        $cacheFile = $cachePath . $id . '.php';

        $app     = Factory::getApplication();
        $doc     = $app->getDocument();
        $wa      = $doc->getWebAssetManager();

        if ($app->isClient('site') && !file_exists($cachePath)) {
            Folder::create($cachePath, 0755);
        }

        // Cache time
        if ($global_cache_time !== '0') {
            // Time for cache files: Component settings
            $time      = $global_cache_time;
        } else {
            // Time for cache files: 60 × 60 × 24 = 24 Hours
            $time      = 60 * 60 * 24;
        }
        // Use cache files
        if (($cache === true && $global_page_cache == 1) && $id && $app->isClient('site') && file_exists($cacheFile) && filemtime($cacheFile) > time() - $time) {
            $cacheContent = file_get_contents($cacheFile);
            // Remove the PHP tag from the cache content
            $cacheContent = str_replace('<?php die("Access Denied"); ?>', '', $cacheContent);

            $cacheCss            = '';
            $cacheScript         = '';
            $cacheScriptTypeJson = '';

            // Find all the <style> tags and extract its content.
            preg_match_all('/<style>(.*?)<\/style>/s', $cacheContent, $matchesStyles);
            if (!empty($matchesStyles[1])) {
                $cacheCss = implode("\n", $matchesStyles[1]);
            }

            // Find all the <script> tags and extract its content.
            preg_match_all('/<script>(.*?)<\/script>/s', $cacheContent, $matchesScripts);
            if (!empty($matchesScripts[1])) {
                $cacheScript = implode("\n", $matchesScripts[1]);
            }

            // Find all the <script type="application/ld+json"> tags and extract its content.
            preg_match_all('/<script\s+type="application\/ld\+json">(.*?)<\/script>/s', $cacheContent, $matchesJson);
            if (!empty($matchesJson[1])) {
                $cacheScriptTypeJson = implode(",", $matchesJson[1]);
            }

            // Add the extracted scripts as inline script
            if (!empty($cacheScript)) {
                $wa->addInlineScript($cacheScript, [], ['type' => 'module']);
            }

            // Add the extracted script type json as inline script
            if (!empty($cacheScriptTypeJson)) {
                $wa->addInlineScript($cacheScriptTypeJson, [], ['type' => 'application/ld+json']);
            }

            // Add the extracted CSS as inline style
            if (!empty($cacheCss)) {
                $wa->addInlineStyle($cacheCss);
            }

            // Return the modified content of the cache file
            return $cacheContent;
        }

        if (is_array($content)) {
            $html    = '';
            $css_all = '';

            foreach ($content as $key => $section) {
                $section                     = SitePageHelper::toArray($section);
                $section_options             = SitePageHelper::escapeHtmlArray($section['options']);
                $title_show                  = ($section_options['title_show'] ?? 1);
                $titleAndSubtitleClasses     = SitePageHelper::getTitleAndSubtitleClasses($section_options);
                $title_class                 = ($titleAndSubtitleClasses['title'] === '') ? '' : ' ' . $titleAndSubtitleClasses['title'];
                $subtitle_class              = ($titleAndSubtitleClasses['subtitle'] === '') ? '' : ' ' . $titleAndSubtitleClasses['subtitle'];
                $subtitleMargin              = '';
                $heading_alignment           = '';
                $position_relative           = '';
                $section_svg_divider_z_index = '';
                $section_css                 = '';
                $section_divider_top         = '';
                $section_divider_bottom      = '';
                $buildStyle                  = '';
                $container_buildStyle        = '';
                $section_full_width          = ($section_options['section_full_width'] ?? 1);
                $section_classes             = '';
                $section_mask_classes        = '';
                $grid_match                  = ($section_options['grid_match'] ?? 1);
                $grid_match_row              = (isset($section_options['grid_match_row']) && $section_options['grid_match_row'] === "1" && $section_options['grid_match'] === "1" ? ' row: false' : '');
                $grid_match_target           = (isset($section_options['grid_match_target']) && $section_options['grid_match'] === "1" && !empty($section_options['grid_match_target']) ? 'target: ' . $section_options['grid_match_target'] . ';' : '');
                $grid_gutter_class           = '';
                $container_classes           = '';
                $section_height_viewport     = '';
                $section_parallax            = '';
                $background_image_srcset     = ($section_options['background_image_srcset'] ?? 0);
                $srcsets                     = [];
                $srcset                      = '';
                $section_style               = [];

                $custom_background = ($section_options['custom_background'] ?? 0);

                $shape_position      = ($section_options['shape_position'] ?? 'left');
                $section_shape_class = '';
                $shape_margin_top    = (!empty($section_options['shape_margin_top']) ? $section_options['shape_margin_top'] . 'px' : 0);
                $shape_margin_right  = (!empty($section_options['shape_margin_right']) ? $section_options['shape_margin_right'] . 'px' : 0);
                $shape_margin_bottom = (!empty($section_options['shape_margin_bottom']) ? $section_options['shape_margin_bottom'] . 'px' : 0);
                $shape_margin_left   = (!empty($section_options['shape_margin_left']) ? $section_options['shape_margin_left'] . 'px' : 0);

                $shape_position_zindex = ($section_options['shape_position_zindex'] ?? 0);
                $shape_border          = (!empty($section_options['shape_border']) ? $section_options['shape_border'] . 'px' : '');
                $shape_transform       = ($section_options['shape_transform'] ?? 'unset');
                $shape_shadow          = (!empty($section_options['shape_shadow']) ? ' ' . $section_options['shape_shadow'] : '');

                $shape_background_image_repeat     = ($section_options['shape_background_image_repeat'] ?? 0);
                $shape_background_image_attachment = ($section_options['shape_background_image_attachment'] ?? "");
                $shape_background_image_position   = (!empty($section_options['shape_background_image_position']) && !empty($section_options['shape_background_image']) ? ' ' . $section_options['shape_background_image_position'] : '');

                $section_svg_background_image_responsive = ($section_options['section_svg_background_image_responsive'] ?? '');

                $section_key_show = ($section_options['section_key'] ?? 0);

                $video_source = '';

                $currentDate              = Factory::getDate()->format('Y-m-d H:i:s');
                $get_publish_down_section = ($section_options['publish_down'] ?? null);
                $get_publish_up_section   = ($section_options['publish_up'] ?? null);

                $publish_down      = $get_publish_down_section == null ? null : $section_options['publish_down'];
                $isExpired         = !is_null($publish_down) && strtotime($publish_down) < strtotime($currentDate);
                $isNotPublishedYet = !is_null($get_publish_up_section) && strtotime($get_publish_up_section) > strtotime($currentDate);
                $isUnpublished     = $isNotPublishedYet || $isExpired;

                $section_show = true;

                if ($isExpired) {
                    $section_show = false;
                } elseif ($isUnpublished) {
                    $section_show = false;
                }


                // Section visibility and check date
                if ($section['admin_visibility'] === "1" && $section_show) {
                    if (!empty($section_options['id'])) {
                        $section_id = ' id="' . $section_options['id'] . '"';
                    } else {
                        $section_id = '';
                    }
                    if (!empty($section_options['class'])) {
                        $section_classes .= ' ' . $section_options['class'];
                    }

                    // Alignment section content
                    if ($section_options['section_align_content'] === "1" && $section_options['heading_alignment'] === 'uk-text-left' && (!empty($section_options['title']) || !empty($section_options['subtitle']))) {
                        $section_classes .= ' uk-flex uk-flex-middle';
                    } elseif ($section_options['section_align_content'] === "1" && $section_options['heading_alignment'] === 'uk-text-right' && (!empty($section_options['title']) || !empty($section_options['subtitle']))) {
                        $section_classes .= ' uk-flex uk-flex-middle uk-flex-right';
                    } elseif ($section_options['section_align_content'] === "1" && $section_options['heading_alignment'] === 'uk-text-center' && (!empty($section_options['title']) || !empty($section_options['subtitle']))) {
                        $section_classes .= ' uk-flex uk-flex-middle uk-flex-center';
                    } elseif ($section_options['section_align_content'] === "1" && $section_full_width === "1" && !empty($section_options['container_width']) && $section_options['section_load_components'] === "1") {
                        $section_classes   .= ' uk-flex uk-flex-middle';
                        $container_classes .= ' uk-width-3-4@xl uk-width-2-3@l';
                    } elseif ($section_options['section_align_content'] === "1") {
                        $section_classes .= ' uk-flex uk-flex-middle';
                    }
                    // END Alignment section content

                    if (!empty($section_options['visibility'])) {
                        $section_classes .= ' ' . $section_options['visibility'];
                    }
                    if (!empty($section_options['hidden'])) {
                        $section_classes .= ' ' . $section_options['hidden'];
                    }
                    if (!empty($section_options['section_padding'])) {
                        $section_classes .= ' ' . $section_options['section_padding'];
                    }
                    if (!empty($section_options['section_margin'])) {
                        $section_classes .= ' ' . $section_options['section_margin'];
                    }

                    // Section Dividers
                    if ((!empty($section_options['section_divider_bottom']) || !empty($section_options['section_divider_top'])) && $section_options['section_divider_z_index'] === "1") {
                        $section_svg_divider_z_index = ' uk-position-z-index';
                    }

                    $divider_paths = PageHelper::getSectionDividers($section_options);

                    if (!empty($section_options['section_divider_top'])) {
                        $svg_paths_top   = $divider_paths['svg_paths_top'];
                        $section_classes .= ' ui-separator-top-' . $key;

                        if (!empty($section_options['section_divider_top_color'])) {
                            $shape_paths_color_t = 'fill: ' . $section_options['section_divider_top_color'] . ';';
                        } elseif (($section_options['section_divider_top_gradient'] === "1") && (!empty($section_options['section_divider_top_color_1'])) && (!empty($section_options['section_divider_top_color_2']))) {
                            $shape_paths_color_t = 'fill: url(#ui-gradient-' . $key . ');';
                        } else {
                            $shape_paths_color_t = '';
                        }

                        if ($section_options['section_divider_top_dropshadow'] === "1") {
                            $section_divider_top .= '<svg class="ui-separator ui-separator-top-' . $key . $section_svg_divider_z_index . '" viewBox="0 0 200 30" preserveAspectRatio="none" xmlns="http://www.w3.org/2000/svg" xmlns:xlink="http://www.w3.org/1999/xlink" style="overflow: visible">';
                        } else {
                            $section_divider_top .= '<svg class="ui-separator ui-separator-top-' . $key . $section_svg_divider_z_index . '" viewBox="0 0 200 30" preserveAspectRatio="none" xmlns="http://www.w3.org/2000/svg" xmlns:xlink="http://www.w3.org/1999/xlink">';
                        }
                        // SVG Filter´s Gradient / DropShadow (Top)
                        if (($section_options['section_divider_top_gradient'] === "1") || ($section_options['section_divider_top_dropshadow'] === "1")) {
                            $section_divider_top .= '<defs>';
                            // Gradient
                            if (($section_options['section_divider_top_gradient'] === "1") && (!empty($section_options['section_divider_top_color_1'])) && (!empty($section_options['section_divider_top_color_2']))) {
                                $section_divider_top .= '<linearGradient id="ui-gradient-' . $key . '" x1="' . $section_options['gradient_x1_position'] . '%" y1="' . $section_options['gradient_y1_position'] . '%" x2="' . $section_options['gradient_x2_position'] . '%" y2="' . $section_options['gradient_y2_position'] . '%" color-interpolation-filters="sRGB">';
                                $section_divider_top .= '<stop offset="' . $section_options['section_divider_top_color_1_position'] . '%" style="stop-color:' . $section_options['section_divider_top_color_1'] . ';stop-opacity:1" />';
                                $section_divider_top .= '<stop offset="' . $section_options['section_divider_top_color_2_position'] . '%" style="stop-color:' . $section_options['section_divider_top_color_2'] . ';stop-opacity:1" />';
                                $section_divider_top .= '</linearGradient>';
                            }
                            // DropShadow
                            if ($section_options['section_divider_top_dropshadow'] === "1") {
                                $section_divider_top .= '<filter id="dropshadowfiltertop-section-' . $key . '" filterUnits="userSpaceOnUse" x="0" y="0" width="100%" height="120%" color-interpolation-filters="sRGB">';
                                $section_divider_top .= '<feGaussianBlur in="SourceAlpha" result="blur" stdDeviation="' . $section_options['section_dropShadow_blur_top'] . '"></feGaussianBlur>';
                                $section_divider_top .= '<feFlood flood-color="' . $section_options['section_filter_dropShadow_color_top'] . '" result="color" ></feFlood>';
                                $section_divider_top .= '<feComposite in="color" in2="blur" operator="in" result="ShadowBlur"></feComposite>';
                                $section_divider_top .= '<feOffset in="ShadowBlur" result="dropShadow" dx="0" dy="' . $section_options['section_dropShadow_blur_y_offset_top'] . '"></feOffset>';
                                $section_divider_top .= '<feMerge>';
                                $section_divider_top .= '<feMergeNode in="dropShadow"></feMergeNode>';
                                $section_divider_top .= '<feMergeNode in="SourceGraphic"></feMergeNode>';
                                $section_divider_top .= '</feMerge>';
                                $section_divider_top .= '</filter>';
                            }
                            $section_divider_top .= '</defs>';
                        }
                        // END SVG Filters
                        if ($section_options['section_divider_top_dropshadow'] === "1") {
                            $section_divider_top .= '<g filter="url(#dropshadowfiltertop-section-' . $key . ')">';
                            $section_divider_top .= $svg_paths_top;
                            $section_divider_top .= '</g>';
                        } else {
                            $section_divider_top .= $svg_paths_top;
                        }

                        $section_divider_top .= '</svg>';

                        // Inline CSS Top Divider
                        if (!empty($section_options['section_divider_top_height'])) {
                            $section_css .= '.ui-separator.ui-separator-top-' . $key . '{height:' . $section_options['section_divider_top_height'] . 'px}';
                        }
                        $section_css .= '.ui-separator.ui-separator-top-' . $key . ' path{' . $shape_paths_color_t . ' stroke-linecap:round}';
                    }

                    if (!empty($section_options['section_divider_bottom'])) {
                        $svg_paths_bottom = $divider_paths['svg_paths_bottom'];
                        $section_classes  .= ' ui-separator-bottom-' . $key;

                        if (!empty($section_options['section_divider_bottom_color'])) {
                            $shape_paths_color_b = 'fill:' . $section_options['section_divider_bottom_color'] . ';';
                        } elseif (($section_options['section_divider_bottom_gradient'] === "1") && (!empty($section_options['section_divider_bottom_color_1'])) && (!empty($section_options['section_divider_bottom_color_2']))) {
                            $shape_paths_color_b = 'fill:url(#ui-gradient-bottom-' . $key . ');';
                        } else {
                            $shape_paths_color_b = '';
                        }

                        if ($section_options['section_divider_bottom_dropshadow'] === "1") {
                            $section_divider_bottom .= '<svg class="ui-separator ui-separator-bottom-' . $key . $section_svg_divider_z_index . '" viewBox="0 0 200 30" preserveAspectRatio="none" xmlns="http://www.w3.org/2000/svg" xmlns:xlink="http://www.w3.org/1999/xlink" style="overflow: visible">';
                        } else {
                            $section_divider_bottom .= '<svg class="ui-separator ui-separator-bottom-' . $key . $section_svg_divider_z_index . '" viewBox="0 0 200 30" preserveAspectRatio="none" xmlns="http://www.w3.org/2000/svg" xmlns:xlink="http://www.w3.org/1999/xlink">';
                        }
                        // SVG Filter´s Gradient / DropShadow (Bottom)
                        if (($section_options['section_divider_bottom_gradient'] === "1") || ($section_options['section_divider_bottom_dropshadow'] === "1")) {
                            $section_divider_bottom .= '<defs>';
                            // Gradient
                            if (($section_options['section_divider_bottom_gradient'] === "1") && (!empty($section_options['section_divider_bottom_color_1'])) && (!empty($section_options['section_divider_bottom_color_2']))) {
                                $section_divider_bottom .= '<linearGradient id="ui-gradient-bottom-' . $key . '" x1="' . $section_options['gradient_x1_position_bottom'] . '%" y1="' . $section_options['gradient_y1_position_bottom'] . '%" x2="' . $section_options['gradient_x2_position_bottom'] . '%" y2="' . $section_options['gradient_y2_position_bottom'] . '%">';
                                $section_divider_bottom .= '<stop offset="' . $section_options['section_divider_bottom_color_1_position'] . '%" style="stop-color:' . $section_options['section_divider_bottom_color_1'] . ';stop-opacity:1" />';
                                $section_divider_bottom .= '<stop offset="' . $section_options['section_divider_bottom_color_2_position'] . '%" style="stop-color:' . $section_options['section_divider_bottom_color_2'] . ';stop-opacity:1" />';
                                $section_divider_bottom .= '</linearGradient>';
                            }
                            // DropShadow
                            if ($section_options['section_divider_bottom_dropshadow'] === "1") {
                                $section_divider_bottom .= '<filter id="dropshadowfilter-bottom-section-' . $key . '" filterUnits="userSpaceOnUse" x="0" y="-20%" width="100%" height="120%" color-interpolation-filters="sRGB">';
                                $section_divider_bottom .= '<feGaussianBlur in="SourceAlpha" result="blur" stdDeviation="' . $section_options['section_dropShadow_blur_bottom'] . '"></feGaussianBlur>';
                                $section_divider_bottom .= '<feFlood flood-color="' . $section_options['section_filter_dropShadow_color_bottom'] . '" result="color" ></feFlood>';
                                $section_divider_bottom .= '<feComposite in="color" in2="blur" operator="in" result="ShadowBlur"></feComposite>';
                                $section_divider_bottom .= '<feOffset in="ShadowBlur" result="dropShadow" dx="0" dy="' . $section_options['section_dropShadow_blur_y_offset_bottom'] . '"></feOffset>';
                                $section_divider_bottom .= '<feMerge>';
                                $section_divider_bottom .= '<feMergeNode in="dropShadow"></feMergeNode>';
                                $section_divider_bottom .= '<feMergeNode in="SourceGraphic"></feMergeNode>';
                                $section_divider_bottom .= '</feMerge>';
                                $section_divider_bottom .= '</filter>';
                            }
                            $section_divider_bottom .= '</defs>';
                        }
                        // END SVG Filters
                        if ($section_options['section_divider_bottom_dropshadow'] === "1") {
                            $section_divider_bottom .= '<g filter="url(#dropshadowfilter-bottom-section-' . $key . ')">';
                            $section_divider_bottom .= $svg_paths_bottom;
                            $section_divider_bottom .= '</g>';
                        } else {
                            $section_divider_bottom .= $svg_paths_bottom;
                        }

                        $section_divider_bottom .= '</svg>';

                        // Inline CSS Bottom Divider
                        if (!empty($section_options['section_divider_bottom_height'])) {
                            $section_css .= '.ui-separator.ui-separator-bottom-' . $key . '{height:' . $section_options['section_divider_bottom_height'] . 'px}';
                        }
                        $section_css .= '.ui-separator.ui-separator-bottom-' . $key . ' path{' . $shape_paths_color_b . 'stroke-linecap:round}';
                    }
                    // END Section Dividers

                    // Section Background
                    if ($custom_background === "1") {
                        if (!empty($section_options['background_image']) || !empty($section_options['background_image_size'])) {
                            // Background srcset
                            if (!empty($section_options['background_image']) && $background_image_srcset === "1" && !empty($section_options['image_width_srcset'])) {
                                [$width, $height] = getimagesize(Uri::root() . $section_options['background_image']);
                                $ex        = '.' . pathinfo($section_options['background_image'], PATHINFO_EXTENSION);
                                $file_name = 'th_' . pathinfo($section_options['background_image'], PATHINFO_FILENAME);
                                // ratio
                                $ratio = round($width / $height * 100);

                                $typeItem         = str_replace('_', '-', $section['type']) . 's';
                                $thumbnail_folder = 'media' . DIRECTORY_SEPARATOR . 'com_uithemebuilderlite' . DIRECTORY_SEPARATOR . 'ui-themebuilder-thumb' . DIRECTORY_SEPARATOR . $typeItem;

                                $tmp_srcset    = explode(',', $section_options['image_width_srcset']);
                                $merge_srcsets = array_merge($srcsets, $tmp_srcset);
                                $srcsets       = array_map('trim', $merge_srcsets);
                                $srcsets       = array_unique($srcsets);

                                $min_val = min($srcsets);

                                $srcset .= ' data-sizes="';
                                $srcset .= '(min-width:  ' . $min_val . 'px) 90vw, ';
                                $srcset .= '(max-width: ' . $width . 'px) 100vw, ';
                                $srcset .= '(max-aspect-ratio: ' . $width . '/' . $height . ') ' . $ratio . 'vh" ';
                                $srcset .= 'data-srcset="';

                                foreach ($srcsets as $i => $val) {
                                    if (is_numeric($val) && $val < 4000) {
                                        unset($srcsets[$i]);
                                        $srcset .= Uri::root() . $thumbnail_folder . '/' . $file_name . '_' . $val . $ex . ' ' . $val . 'w, ';
                                    }
                                }

                                $srcset .= Uri::root() . $section_options['background_image'] . ' ' . $width . 'w" ';
                                $srcset .= 'data-src="' . Uri::root() . $section_options['background_image'] . '"';
                                $srcset .= ' uk-img=""';
                            } elseif (!empty($section_options['background_image'])) {
                                $section_style['background-image'] = Uri::root() . $section_options['background_image'];
                            } else {
                                $section_style['background-image'] = 'https://media.ui-themebuilder.com/com_uithemebuilderpro/unsplash/unsplash_custom.jpg?' . time();
                            }

                            if (!empty($section_options['background_image_responsive']) && !empty($section_options['background_image'])) {
                                $section_classes .= ' ' . $section_options['background_image_responsive'];
                            }
                            if ($section_options['background_image_repeat'] === "0") {
                                $section_classes .= ' uk-background-norepeat';
                            }
                            if (!empty($section_options['background_image_size'])) {
                                $section_classes .= ' ' . $section_options['background_image_size'];
                            }

                            // Section Parallax Background Settings
                            if ($section_options['parallax_animation_background'] === "1") {
                                if (!empty($section_options['parallax_animation_media'])) {
                                    $parallax_attr_media = ' media: ' . $section_options['parallax_animation_media'] . ';';
                                } else {
                                    $parallax_attr_media = '';
                                }

                                if ($section_options['parallax_animation_background_viewport']) {
                                    $parallax_animation_background_viewport = ' viewport: ' . $section_options['parallax_animation_background_viewport'] . ';';
                                } else {
                                    $parallax_animation_background_viewport = '';
                                }

                                if (!empty($section_options['parallax_animation_background_x'])) {
                                    $parallax_x = 'bgx: ' . $section_options['parallax_animation_background_x'] . ';';
                                } else {
                                    $parallax_x = '';
                                }
                                if (!empty($section_options['parallax_animation_background_y'])) {
                                    $parallax_y = 'bgy: ' . $section_options['parallax_animation_background_y'] . ';';
                                } else {
                                    $parallax_y = '';
                                }

                                // Filters
                                if (!empty($section_options['parallax_animation_background_grayscale'])) {
                                    $parallax_animation_background_grayscale = ' grayscale: ' . $section_options['parallax_animation_background_grayscale'] . ';';
                                } else {
                                    $parallax_animation_background_grayscale = '';
                                }
                                if (!empty($section_options['parallax_animation_background_saturate'])) {
                                    $parallax_animation_background_saturate = ' saturate: ' . $section_options['parallax_animation_background_saturate'] . ';';
                                } else {
                                    $parallax_animation_background_saturate = '';
                                }
                                if (!empty($section_options['parallax_animation_background_color']) && !empty($section_options['background_color'])) {
                                    $parallax_animation_background_color = ' background-color: ' . $section_options['background_color'] . ',' . $section_options['parallax_animation_background_color'] . ';';
                                } else {
                                    $parallax_animation_background_color = '';
                                }

                                $section_parallax = ' uk-parallax="' . $parallax_x . $parallax_y . $parallax_attr_media . $parallax_animation_background_viewport . $parallax_animation_background_grayscale . $parallax_animation_background_saturate . $parallax_animation_background_color . '"';
                            }
                        }

                        if (!empty($section_options['background_color'])) {
                            $section_style['background-color'] = $section_options['background_color'];
                        }
                        if (!empty($section_options['background_color']) && !empty($section_options['background_image_blend_modes'])) {
                            $section_classes .= ' ' . $section_options['background_image_blend_modes'];
                        }

                        if (!empty($section_options['background_image']) && !empty($section_options['background_image_attachment'])) {
                            $section_classes .= ' ' . $section_options['background_image_attachment'];
                        }
                        if (!empty($section_options['background_image']) && !empty($section_options['background_image_position'])) {
                            $section_classes .= ' ' . $section_options['background_image_position'];
                        }
                    }
                    // Gradient Background
                    elseif ($custom_background === "2") {
                        $color1_section = ($section_options['gradient_color_1'] ?? '');
                        $color2_section = ($section_options['gradient_color_2'] ?? '');
                        $color3_section = ($section_options['gradient_color_3'] ?? '');

                        $color1_position = (isset($section_options['gradient_color_1_position']) ? $section_options['gradient_color_1_position'] . '%' : '');
                        $color2_position = (isset($section_options['gradient_color_2_position']) ? $section_options['gradient_color_2_position'] . '%' : '');
                        $color3_position = (isset($section_options['gradient_color_3_position']) ? $section_options['gradient_color_3_position'] . '%' : '');

                        $radial_conical_x_position = (isset($section_options['radial_conical_x_position']) ? $section_options['radial_conical_x_position'] . '%' : '');
                        $radial_conical_y_position = (isset($section_options['radial_conical_y_position']) ? $section_options['radial_conical_y_position'] . '%' : '');

                        // linear gradient
                        if ($section_options['gradient_effect'] === 'linear' && (!empty($color1_section) || !empty($color2_section) || !empty($color3_section))) {
                            $section_style['background-color'] = $color1_section; // Fallback color
                            // 3 active colors
                            if (!empty($color1_section) && !empty($color2_section) && !empty($color3_section)) {
                                $section_style['background'] = 'linear-gradient(' . $section_options['gradient_angle'] . 'deg, ' . $color1_section . ' ' . $color1_position . ', ' . $color2_section . ' ' . $color2_position . ', ' . $color3_section . ' ' . $color3_position . ')';
                                // 2 active colors
                            } elseif (!empty($color1_section) && !empty($color2_section)) {
                                $section_style['background'] = 'linear-gradient(' . $section_options['gradient_angle'] . 'deg,' . $color1_section . $color1_position . ', ' . $color2_section . $color2_position . ')';
                            }
                            $section_style['background-attachment'] = 'scroll';

                            // radial gradient
                        } elseif ($section_options['gradient_effect'] === 'radial' && ((!empty($color1_section) && !empty($color2_section)) || !empty($color3_section)) && ($radial_conical_x_position && $radial_conical_y_position)) {
                            $section_style['background-color'] = $color1_section; // Fallback color
                            // 3 active colors
                            if (!empty($color1_section) && !empty($color2_section) && !empty($color3_section)) {
                                $section_style['background'] = 'radial-gradient(circle at ' . $radial_conical_x_position . ' ' . $radial_conical_y_position . ', ' . $color1_section . ' ' . $color1_position . ', ' . $color2_section . ' ' . $color2_position . ', ' . $color3_section . ' ' . $color3_position . ')';
                                // 2 active colors
                            } elseif (!empty($color1_section) && !empty($color2_section)) {
                                $section_style['background'] = 'radial-gradient(circle at ' . $radial_conical_x_position . ' ' . $radial_conical_y_position . ', ' . $color1_section . ' ' . $color1_position . ', ' . $color2_section . ' ' . $color2_position . ')';
                            }
                            $section_style['background-attachment'] = 'scroll';

                            // conical gradient
                        } elseif ($section_options['gradient_effect'] === 'conical' && ((!empty($color1_section) && !empty($color2_section)) || !empty($color3_section)) && ($radial_conical_x_position && $radial_conical_y_position)) {
                            $section_style['background-color'] = $color1_section; // Fallback color
                            // 3 active colors
                            if (!empty($color1_section) && !empty($color2_section) && !empty($color3_section)) {
                                $section_style['background'] = 'conic-gradient(from ' . $section_options['gradient_angle'] . 'deg at ' . $radial_conical_x_position . ' ' . $radial_conical_y_position . ', ' . $color1_section . ' ' . $color1_position . ', ' . $color2_section . ' ' . $color2_position . ', ' . $color3_section . ' ' . $color3_position . ')';
                                // 2 active colors
                            } elseif (!empty($color1_section) && !empty($color2_section)) {
                                $section_style['background'] = 'conic-gradient(from ' . $section_options['gradient_angle'] . 'deg at ' . $radial_conical_x_position . ' ' . $radial_conical_y_position . ', ' . $color1_section . ' ' . $color1_position . ', ' . $color2_section . ' ' . $color2_position . ')';
                            }
                        }
                    }
                    // SVG Background
                    elseif ($custom_background === "3") {
                        if (!empty($section_options['section_svg_list'])) {
                            $svg_path                   = Uri::root() . $section_options['section_svg_list'];
                            $color_placeholder_1        = $section_options['section_svg_background_color_fallback'];
                            $color_placeholder_2        = $section_options['section_svg_background_color_2'];
                            $color_placeholder_3        = $section_options['section_svg_background_color_3'];
                            $text_placeholder           = $section_options['section_svg_background_text'];
                            $text_font_size_placeholder = $section_options['section_svg_background_text_font_size'] . 'px';
                            $opacity_placeholder        = $section_options['section_svg_background_image_opacity'];

                            $section_classes .= ' ui-section-svg-bg-' . $key;
                            $section_classes .= ' ' . $section_svg_background_image_responsive;

                            // CSS output
                            $section_css .= '.ui-section-svg-bg-' . $key . '{';
                            if (!empty($section_options['section_svg_background_color_fallback'])) {
                                $section_css .= 'background: ' . $section_options['section_svg_background_color_fallback'] . ';';
                                $section_css .= 'background-color: ' . $section_options['section_svg_background_color_fallback'] . ';';
                            }
                            if (!empty($section_options['section_svg_background_attachment'])) {
                                $section_css .= 'background-attachment: ' . $section_options['section_svg_background_attachment'] . ';';
                            }
                            if (!empty($section_options['section_svg_background_image_size'])) {
                                $section_css .= 'background-size: ' . $section_options['section_svg_background_image_size'] . ';';
                            }
                            if (!empty($section_options['section_svg_background_image_repeat'])) {
                                $section_css .= 'background-repeat: ' . $section_options['section_svg_background_image_repeat'] . ';';
                            }
                            if (!empty($section_options['section_svg_background_position_x']) && !empty($section_options['section_svg_background_position_y'])) {
                                $section_css .= 'background-position: ' . $section_options['section_svg_background_position_x'] . ' ' . $section_options['section_svg_background_position_y'] . ';';
                            }
                            $section_css .= 'background-image: url("data:image/svg+xml,' . SitePageHelper::convertSVGtoURI($svg_path, $key, $color_placeholder_1, $color_placeholder_2, $color_placeholder_3, $text_placeholder, $text_font_size_placeholder, $opacity_placeholder) . '");';
                            $section_css .= '}';
                        }
                    }
                    // Shape Section Background
                    elseif ($custom_background === "4") {
                        if (!empty($section_options['background_color']) && empty($section_options['shape_background'])) {
                            $section_style['background-color'] = $section_options['background_color'];
                        }

                        // default section background
                        if (!empty($section_options['shape_background'])) {
                            $section_classes .= ' ' . $section_options['shape_background'];
                        }

                        // square / circle shape
                        if (($section_options['shape'] === 'square' || $section_options['shape'] === 'circle') && ((!empty($section_options['shape_width']) && !empty($section_options['shape_height'])) || (!empty($section_options['shape_width_pixel']) && !empty($section_options['shape_height_pixel'])))) {
                            // CSS Class
                            if ($section_options['shape'] === 'square') {
                                $section_shape_class = 'ui-shape-square-' . $key;
                            } else {
                                $section_shape_class = 'ui-shape-circle-' . $key;
                            }
                            // CSS
                            $section_css .= '.' . $section_shape_class . '{';
                            // width & height pixel or %
                            if (!empty($section_options['shape_width']) && !empty($section_options['shape_height'])) {
                                $section_css .= 'width: ' . $section_options['shape_width'] . '%;';
                                $section_css .= 'height: ' . $section_options['shape_height'] . '%;';
                            } elseif (!empty($section_options['shape_width_pixel']) && !empty($section_options['shape_height_pixel'])) {
                                $section_css .= 'width: ' . $section_options['shape_width_pixel'] . 'px;';
                                $section_css .= 'height: ' . $section_options['shape_height_pixel'] . 'px;';
                            }
                            // background-color
                            if (!empty($section_options['shape_color'])) {
                                $section_css .= 'background-color: ' . $section_options['shape_color'] . ';';
                            } else {
                                $section_css .= 'background-color: transparent;';
                            }
                            // background-image
                            if (!empty($section_options['shape_background_image'])) {
                                $section_css .= 'background-image: url("' . Uri::root() . $section_options['shape_background_image'] . '");';
                                // image repeat
                                if ($shape_background_image_repeat === "0") {
                                    $section_css .= 'background-repeat: no-repeat;';
                                }
                                // image size
                                if (!empty($section_options['shape_background_image_size'])) {
                                    $section_css .= 'background-size: ' . $section_options['shape_background_image_size'] . ';';
                                }
                                // image blend modes
                                if (!empty($section_options['shape_background_image_blend_modes']) && !empty($section_options['shape_color'])) {
                                    $section_css .= 'background-blend-mode: ' . $section_options['shape_background_image_blend_modes'] . ';';
                                }
                                // image attachment
                                if ($shape_background_image_attachment === "fixed") {
                                    $section_css .= 'background-attachment: fixed;';
                                }
                            }
                            // position
                            $section_css .= 'top: 0;';
                            if ($shape_position === 'right') {
                                $section_css .= 'right: 0;';
                            } else {
                                $section_css .= 'left: 0;';
                            }
                            // margin
                            if (!$shape_margin_top || !$shape_margin_right || !$shape_margin_bottom || !$shape_margin_left) {
                                $section_css .= 'margin: ' . $shape_margin_top . ' ' . $shape_margin_right . ' ' . $shape_margin_bottom . ' ' . $shape_margin_left . ';';
                            }
                            // border
                            if ($shape_border && !empty($section_options['shape_border_style']) && !empty($section_options['shape_border_color'])) {
                                $section_css .= 'border: ' . $shape_border . ' ' . $section_options['shape_border_style'] . ' ' . $section_options['shape_border_color'] . ';';
                            }
                            if ($section_options['shape'] === 'circle') {
                                $section_css .= 'border-radius: 50%;';
                            }
                            // transform rotate
                            if ($shape_transform === 'rotate' && !empty($section_options['shape_transform_rotate'])) {
                                $section_css .= 'transform: rotate(' . $section_options['shape_transform_rotate'] . 'deg);';
                            }
                            // transform rotate3d
                            elseif ($shape_transform === 'rotate3d' && !empty($section_options['shape_transform_rotate']) && !empty($section_options['shape_transform3d_1']) && !empty($section_options['shape_transform3d_2']) && !empty($section_options['shape_transform3d_3'])) {
                                $section_css .= 'transform: rotate3d(' . $section_options['shape_transform3d_1'] . ', ' . $section_options['shape_transform3d_2'] . ', ' . $section_options['shape_transform3d_3'] . ', ' . $section_options['shape_transform_rotate'] . 'deg);';
                            } else {
                                $section_css .= 'transform: unset;';
                            }
                            // z-index
                            if ($shape_position_zindex === "1") {
                                $section_css .= 'z-index: 1;';
                            }
                            $section_css .= '}';
                        }
                        // triangle shape
                        elseif ($section_options['shape'] === 'triangle' && !empty($section_options['shape_color'])) {
                            // CSS Class
                            $section_shape_class = 'ui-shape-triangle-' . $key;

                            // CSS
                            $section_css .= '.' . $section_shape_class . '{';
                            // width & height
                            $section_css .= 'width: 0;';
                            $section_css .= 'height: 0;';

                            // background-color
                            $section_css .= 'background-color: transparent;';
                            // position
                            $section_css .= 'top: 0;';
                            if ($shape_position === 'right') {
                                $section_css .= 'right: 0;';
                            } else {
                                $section_css .= 'left: 0;';
                            }
                            // margin
                            if (!$shape_margin_top || !$shape_margin_right || !$shape_margin_bottom || !$shape_margin_left) {
                                $section_css .= 'margin: ' . $shape_margin_top . ' ' . $shape_margin_right . ' ' . $shape_margin_bottom . ' ' . $shape_margin_left . ';';
                            }
                            // border (build triangle)
                            if (!empty($section_options['shape_height_pixel'])) {
                                $section_css .= 'border-left: ' . $section_options['shape_height_pixel'] / 2 . 'px solid transparent;';
                                $section_css .= 'border-right: ' . $section_options['shape_height_pixel'] / 2 . 'px solid transparent;';
                                $section_css .= 'border-bottom: ' . $section_options['shape_height_pixel'] . 'px solid ' . $section_options['shape_color'] . ';';
                            }
                            // transform rotate
                            if ($shape_transform === 'rotate' && !empty($section_options['shape_transform_rotate'])) {
                                $section_css .= 'transform: rotate(' . $section_options['shape_transform_rotate'] . 'deg);';
                                // transform rotate3d
                            } elseif ($shape_transform === 'rotate3d' && !empty($section_options['shape_transform_rotate']) && !empty($section_options['shape_transform3d_1']) && !empty($section_options['shape_transform3d_2']) && !empty($section_options['shape_transform3d_3'])) {
                                $section_css .= 'transform: rotate3d(' . $section_options['shape_transform3d_1'] . ', ' . $section_options['shape_transform3d_2'] . ', ' . $section_options['shape_transform3d_3'] . ', ' . $section_options['shape_transform_rotate'] . 'deg);';
                            } else {
                                $section_css .= 'transform: unset;';
                            }
                            // z-index
                            if ($shape_position_zindex === "1") {
                                $section_css .= 'z-index: 1;';
                            } else {
                                $section_css .= 'z-index: 0;';
                            }
                            $section_css .= '}';
                        }
                    }
                    // Video Background
                    elseif ($custom_background === "5") {
                        $section_video_classes   = ' ui-video-section-' . $key;
                        $section_video_css_class = trim($section_video_classes);

                        $video_extensions = array ('asf', 'avi', 'flv', 'm4v', 'mp4', 'ogg', 'ogv', 'webm');

                        $video_auto_play        = ($section_options['video_auto_play'] ?? 1);
                        $video_show_controls    = ($section_options['video_show_controls'] ?? 1);
                        $video_show_suggestions = ($section_options['video_show_suggestions'] ?? 1);
                        $video_loop             = ($section_options['video_loop'] ?? 1);
                        $video_show_info        = ($section_options['video_show_info'] ?? 1);
                        $video_muted            = ($section_options['video_muted'] ?? 1);

                        // CSS output
                        $section_css .= '.' . $section_video_css_class . ' {';

                        $section_css .= 'top: 0;';
                        $section_css .= 'left: 0;';
                        $section_css .= 'z-index: 0;';

                        $section_css .= '}';

                        $video_url_tmp = ($section_options['video_url'] ?? 'https://www.youtube-nocookie.com/embed/YE7VzlLtp-4');
                        $video_url     = parse_url($video_url_tmp);

                        // Video autoplay params
                        if ($video_auto_play === "1") {
                            $video_params_autoplay = 'autoplay=1&amp;';
                            $ui_video              = '';
                        } else {
                            $video_params_autoplay = '';
                            $ui_video              = 'autoplay: inview';
                        }
                        // Video controls
                        if ($video_show_controls === "1") {
                            $video_params_controls = 'controls=1';
                        } else {
                            $video_params_controls = 'controls=0';
                        }
                        // Video suggestions
                        if ($video_show_suggestions === "1") {
                            $video_params_rel = 'rel=1';
                        } else {
                            $video_params_rel = 'rel=0';
                        }
                        // Video loop
                        if ($video_loop === "1") {
                            $video_params_loop = 'loop=1';
                        } else {
                            $video_params_loop = 'loop=0';
                        }
                        // Video info
                        if ($video_show_info === "1") {
                            $video_params_show_info = 'showinfo=1';
                        } else {
                            $video_params_show_info = 'showinfo=0';
                        }
                        // Video muted
                        if ($video_muted === "1") {
                            $muted           = 'automute: true';
                            $video_muted_url = '&amp;mute=1';
                        } else {
                            $video_muted_url = '&amp;mute=0';
                            $muted           = '&amp;mute=0';
                        }
                        // Video start at
                        if (!empty($section_options['video_start'])) {
                            $video_start = '&amp;start=' . $section_options['video_start'];
                        } else {
                            $video_start = '';
                        }
                        // Video iframe source
                        if (!empty($section_options['video_url'])) {
                            switch ($video_url['host']) {
                                case 'www.youtube-nocookie.com':
                                case 'youtube.com':
                                case 'www.youtube.com':
                                    $video_class = ' ui-youtube-video';
                                    $video_src   = $section_options['video_url'] . '?' . $video_params_autoplay . $video_params_controls . '&amp;' . $video_params_loop . '&amp;' . $video_params_show_info . '&amp;' . $video_params_rel . '&amp;iv_load_policy=3&amp;showsearch=0&amp;modestbranding=1' . $video_muted_url . '&amp;origin=' . Uri::root() . '&amp;playsinline=1' . $video_start;

                                    $video_source = '<div class="uk-visible@m uk-cover-container uk-position-absolute uk-width-1-1 uk-height-1-1 ' . $section_video_classes . $video_class . '" itemprop="video" itemscope="" itemtype="https://schema.org/VideoObject" uk-scrollspy="cls: uk-animation-fade; target: > *; delay: 250;">';
                                    if (!empty($section_options['thumbnail_image'])) {
                                        $video_source .= '<meta itemprop="thumbnailURL" content="' . Uri::root() . $section_options['thumbnail_image'] . '" />';
                                    }
                                    $video_source .= '<meta itemprop="embedURL" content="' . $section_options['video_url'] . '" />';
                                    $video_source .= '<iframe src="' . $video_src . '" frameborder="0" allow="accelerometer; encrypted-media; gyroscope; picture-in-picture" allowfullscreen="" uk-responsive="" uk-cover="" uk-video="' . $ui_video . $muted . '"></iframe>';
                                    $video_source .= '</div>';

                                    break;

                                case 'vimeo.com':
                                case 'www.vimeo.com':
                                    // $doc->addScript('https://player.vimeo.com/api/player.js');
                                    if ($video_auto_play === "0") {
                                        $vimeoID = '';

                                        if (preg_match("/(https?:\/\/)?(www\.)?(player\.)?vimeo\.com\/?(showcase\/)*([0-9)([a-z]*\/)*([0-9]{6,11})[?]?.*/", $section_options['video_url'], $output_array)) {
                                            $vimeoID = $output_array[6];
                                        }

                                        $js =
                                            '
												$(document).ready(function() { 

													uiVimeoPlayer' . $key . ' = uiVimeoPlayerLoadVideo("vimeoSection' . $key . '",' . $vimeoID . ');
													uiVimeoPlayer' . $key . '.pause();
													$(window).scroll(function(){ uiVimeoPlayerPlayPause("vimeoSection' . $key . '" , uiVimeoPlayer' . $key . '); });

														$("#btnPause").bind("click", function() {
															uiVimeoPlayer' . $key . '.pause();
														});
														$("#btnPlay").bind("click", function() {
															uiVimeoPlayer' . $key . '.play();
														});
												});
											';

                                        $doc->addScriptDeclaration(trim(preg_replace('/^\s+|\n|\r|\s+$/m', '', $js)));
                                    }

                                    $video_class = ' ui-vimeo-video';
                                    $video_src   = 'https://player.vimeo.com/video/' . basename($video_url['path']) . '?' . $video_params_autoplay . $video_params_loop . '&amp;title=0&amp;byline=0&amp;portrait=0';

                                    $video_source = '<div id="vimeoSection' . $key . '" class="uk-visible@m uk-cover-container uk-position-absolute uk-width-1-1 uk-height-1-1 ' . $section_video_classes . $video_class . '" itemprop="video" itemscope="" itemtype="https://schema.org/VideoObject" uk-scrollspy="cls: uk-animation-fade; target: > *; delay: 250;">';
                                    if (!empty($section_options['thumbnail_image'])) {
                                        $video_source .= '<meta itemprop="thumbnailURL" content="' . Uri::root() . $section_options['thumbnail_image'] . '" />';
                                    }
                                    $video_source .= '<meta itemprop="embedURL" content="' . $section_options['video_url'] . '" />';
                                    $video_source .= '<iframe src="' . $video_src . '" frameborder="0" allow="accelerometer; encrypted-media; gyroscope; picture-in-picture" allowfullscreen="" uk-responsive="" uk-cover="" uk-video="' . $muted . '"></iframe>';
                                    $video_source .= '</div>';

                                    break;

                                default:
                                    if (in_array(File::getExt($section_options['video_url']), $video_extensions)) {
                                        $video_class = ' default-video-player';
                                        // Video autoplay params
                                        if ($video_auto_play === "1") {
                                            $video_params_autoplay = ' autoplay ';
                                            $ui_video              = '';
                                        } else {
                                            $video_params_autoplay = ' ';
                                            $ui_video              = 'autoplay: inview';
                                        }
                                        // Video controls
                                        if ($video_show_controls === "1") {
                                            $video_params_controls = 'controls ';
                                        } else {
                                            $video_params_controls = '';
                                        }
                                        // Video loop
                                        if ($video_loop === "1") {
                                            $video_params_loop = 'loop ';
                                        } else {
                                            $video_params_loop = '';
                                        }
                                        // Video muted
                                        if ($video_muted === "1") {
                                            $muted = 'muted';
                                        } else {
                                            $muted = '';
                                        }

                                        $video_src = $section_options['video_url'];

                                        $video_source = '<div class="uk-visible@m uk-cover-container uk-position-absolute uk-width-1-1 uk-height-1-1 ' . $section_video_classes . $video_class . '" itemprop="video" itemscope="" itemtype="https://schema.org/VideoObject" uk-scrollspy="cls: uk-animation-fade; target: > *; delay: 250;">';
                                        if (!empty($section_options['thumbnail_image'])) {
                                            $video_source .= '<meta itemprop="thumbnailURL" content="' . Uri::root() . $section_options['thumbnail_image'] . '" />';
                                        }
                                        $video_source .= '<meta itemprop="embedURL" content="' . $section_options['video_url'] . '" />';
                                        $video_source .= '<video ' . $video_params_controls . $video_params_loop . $muted . ' playsinline' . $video_params_autoplay . 'uk-cover uk-video="' . $ui_video . '">';
                                        $video_source .= '<source src="' . $video_src . '" type="video/' . File::getExt($section_options['video_url']) . '">';
                                        $video_source .= '</video>';
                                        $video_source .= '</div>';
                                    }
                                    break;
                            }
                        }
                    }
                    // Default Section Background
                    elseif (!empty($section_options['background'])) {
                        $section_classes .= ' ' . $section_options['background'];
                    }

                    // Section Mask Settings
                    $use_svg_custom_mask_size       = ($section_options['use_svg_custom_mask_size'] ?? 0);
                    $svg_mask_fill_color            = ($section_options['svg_mask__color'] ?? '#FFFFFF');
                    $svg_mask_mirrored_horizontally = ($section_options['svg_mask_mirrored_horizontally'] ?? 0);

                    if (!empty($section_options['svg_mask_list']) && ($custom_background === "1" || $custom_background === "2" || $custom_background === "3" || $custom_background === "5")) {
                        $section_mask_classes   = ' ui-mask-section-' . $key;
                        $section_mask_css_class = trim($section_mask_classes);
                        $svg_mask_path          = Uri::root() . 'media/com_uithemebuilderlite/images/svg-masks/' . $section_options['svg_mask_list'];

                        if (empty($section_options['background_color']) && (!empty($section_options['svg_mask__color']))) {
                            $section_style['background-color'] = $svg_mask_fill_color;
                        }
                        // CSS output
                        $section_css .= '.' . $section_mask_css_class . ' {';
                        $section_css .= 'top: 0;';
                        $section_css .= 'left: 0;';
                        $section_css .= 'background-image: url("data:image/svg+xml,' . SitePageHelper::convertSVGtoURI($svg_mask_path, $key, $svg_mask_fill_color, false, false, false, false, false) . '");';
                        $section_css .= 'background-repeat: ' . (isset($section_options['svg_mask_repeat']) ? $section_options['svg_mask_repeat'] . ';' : 'inherit;');
                        if ($use_svg_custom_mask_size === "0") {
                            $section_css .= 'background-size: ' . (isset($section_options['svg_mask_size']) ? $section_options['svg_mask_size'] . ';' : 'auto;');
                        } else {
                            $section_css .= 'background-size: ' . (isset($section_options['svg_custom_mask_size']) ? $section_options['svg_custom_mask_size'] . ';' : '100%;');
                        }
                        if ($svg_mask_mirrored_horizontally === "1") {
                            $section_css .= '-webkit-transform: scaleX(-1);';
                            $section_css .= 'transform: scaleX(-1);';
                        }

                        $section_css .= 'background-position: ' . (isset($section_options['svg_mask_position']) ? $section_options['svg_mask_position'] . ';' : 'center;');

                        $section_css .= '}';
                    }

                    // build section styles
                    if (!empty($section_style)) {
                        $buildStyle = SitePageHelper::buildStyle($section_style);
                    }

                    // Section Container Position relative
                    if ((!empty($section_options['section_divider_bottom']) || !empty($section_options['section_divider_top'])) || ($custom_background === "4" && $section_options['shape'] !== '') || $section_key_show === "1" || !empty($section_options['svg_mask_list'])) {
                        $container_classes .= ' uk-position-relative';
                        $position_relative = ' uk-position-relative';
                    }

                    // Add Inline CSS SVGs Dividers / SVG Background / Shapes / Masks
                    if (!empty($section_css)) {
                        $css_all .= preg_replace('/\s+/', ' ', $section_css);
                    }
                    // END Section Backgrounds

                    if ($grid_match === "1") {
                        $grid_match_attr = ' uk-height-match="' . $grid_match_target . $grid_match_row . '"';
                    } else {
                        $grid_match_attr = '';
                    }

                    if (!empty($section_options['grid_gutter'])) {
                        $grid_gutter_class .= ' ' . $section_options['grid_gutter'];
                    }
                    if (!empty($section_options['grid_row_gap'])) {
                        $grid_gutter_class .= ' ' . $section_options['grid_row_gap'];
                    }
                    if (!empty($section_options['grid_divider'])) {
                        $grid_gutter_class .= ' ' . $section_options['grid_divider'];
                    }

                    if ($section_options['section_height_viewport'] === "1") {
                        if (!empty($section_options['section_height_viewport_offset_top'])) {
                            $section_height_viewport_offset_top = 'offset-top:' . $section_options['section_height_viewport_offset_top'] . ';';
                        } else {
                            $section_height_viewport_offset_top = '';
                        }
                        if (!empty($section_options['section_height_viewport_offset_bottom'])) {
                            $section_height_viewport_offset_bottom = ' offset-bottom:' . $section_options['section_height_viewport_offset_bottom'] . ';';
                        } else {
                            $section_height_viewport_offset_bottom = '';
                        }
                        if (!empty($section_options['section_height_viewport_expand'])) {
                            $section_height_viewport_expand = ' expand:' . $section_options['section_height_viewport_expand'] . ';';
                        } else {
                            $section_height_viewport_expand = '';
                        }
                        if (!empty($section_options['section_height_viewport_min_height'])) {
                            $section_height_viewport_offset_min_height = ' min-height:' . $section_options['section_height_viewport_min_height'] . ';';
                        } else {
                            $section_height_viewport_offset_min_height = '';
                        }
                        $section_height_viewport = ' uk-height-viewport="' . $section_height_viewport_offset_top . $section_height_viewport_offset_bottom . $section_height_viewport_expand . $section_height_viewport_offset_min_height . '"';
                    }

                    if (!empty($section_options['color_style'])) {
                        $section_classes .= ' ' . $section_options['color_style'];
                    }

                    // Scrollspy Animation
                    if ($section_options['scrollspy_animation'] === "1") {
                        $cls = '';
                        if (!empty($section_options['scrollspy_animation_type'])) {
                            $cls = 'cls: ' . $section_options['scrollspy_animation_type'] . ';';
                        }
                        if ($section_options['scrollspy_animation_repeat'] === "1") {
                            $repeat = ' repeat: true';
                        } else {
                            $repeat = '';
                        }
                        if (!empty($section_options['scrollspy_animation_delay'])) {
                            $delay = ' delay: ' . $section_options['scrollspy_animation_delay'] . ';';
                        } else {
                            $delay = ' delay: 500;';
                        }
                        $scrollspy_animation_attr = ' uk-scrollspy="' . $cls . $delay . $repeat . '"';
                    } else {
                        $scrollspy_animation_attr = '';
                    }
                    // END Scrollspy Animation


                    // Parallax Animation
                    // Title
                    if ($section_options['parallax_animation_title'] === "1") {
                        if (!empty($section_options['parallax_animation_media'])) {
                            $parallax_attr_media = ' media: ' . $section_options['parallax_animation_media'] . ';';
                        } else {
                            $parallax_attr_media = '';
                        }

                        if (!empty($section_options['parallax_animation_title_opacity_values'])) {
                            $parallax_attr_title_opacity = 'opacity: ' . $section_options['parallax_animation_title_opacity_values'] . ';';
                        } else {
                            $parallax_attr_title_opacity = '';
                        }

                        if (!empty($section_options['parallax_animation_title_y_values'])) {
                            $parallax_attr_title_y = ' y: ' . $section_options['parallax_animation_title_y_values'] . ';';
                        } else {
                            $parallax_attr_title_y = '';
                        }

                        if (!empty($section_options['parallax_animation_title_x_values'])) {
                            $parallax_attr_title_x = ' x: ' . $section_options['parallax_animation_title_x_values'] . ';';
                        } else {
                            $parallax_attr_title_x = '';
                        }

                        if (!empty($section_options['parallax_animation_title_scale_values'])) {
                            $parallax_attr_title_scale = ' scale: ' . $section_options['parallax_animation_title_scale_values'] . ';';
                        } else {
                            $parallax_attr_title_scale = '';
                        }

                        if ($section_options['parallax_animation_title_viewport']) {
                            $parallax_attr_title_viewport = ' viewport: ' . $section_options['parallax_animation_title_viewport'] . ';';
                        } else {
                            $parallax_attr_title_viewport = '';
                        }

                        $parallax_attr_title = ' uk-parallax="' . $parallax_attr_title_opacity . $parallax_attr_title_y . $parallax_attr_title_x . $parallax_attr_title_scale . $parallax_attr_media . $parallax_attr_title_viewport . '"';
                    } else {
                        $parallax_attr_title = '';
                    }

                    // SubTitle
                    if ($section_options['parallax_animation_subtitle'] === "1") {
                        if (!empty($section_options['parallax_animation_media'])) {
                            $parallax_attr_subtitle_media = ' media: ' . $section_options['parallax_animation_media'] . ';';
                        } else {
                            $parallax_attr_subtitle_media = '';
                        }

                        if (!empty($section_options['parallax_animation_subtitle_opacity_values'])) {
                            $parallax_attr_subtitle_opacity = 'opacity: ' . $section_options['parallax_animation_subtitle_opacity_values'] . ';';
                        } else {
                            $parallax_attr_subtitle_opacity = '';
                        }

                        if (!empty($section_options['parallax_animation_subtitle_y_values'])) {
                            $parallax_attr_subtitle_y = ' y: ' . $section_options['parallax_animation_subtitle_y_values'] . ';';
                        } else {
                            $parallax_attr_subtitle_y = '';
                        }

                        if (!empty($section_options['parallax_animation_subtitle_x_values'])) {
                            $parallax_attr_subtitle_x = ' x: ' . $section_options['parallax_animation_subtitle_x_values'] . ';';
                        } else {
                            $parallax_attr_subtitle_x = '';
                        }

                        if (!empty($section_options['parallax_animation_subtitle_scale_values'])) {
                            $parallax_attr_subtitle_scale = ' scale: ' . $section_options['parallax_animation_subtitle_scale_values'] . ';';
                        } else {
                            $parallax_attr_subtitle_scale = '';
                        }

                        if ($section_options['parallax_animation_subtitle_viewport']) {
                            $parallax_attr_subtitle_viewport = ' viewport: ' . $section_options['parallax_animation_subtitle_viewport'] . ';';
                        } else {
                            $parallax_attr_subtitle_viewport = '';
                        }

                        $parallax_attr_subtitle = ' uk-parallax="' . $parallax_attr_subtitle_opacity . $parallax_attr_subtitle_y . $parallax_attr_subtitle_x . $parallax_attr_subtitle_scale . $parallax_attr_subtitle_media . $parallax_attr_subtitle_viewport . '"';
                    } else {
                        $parallax_attr_subtitle = '';
                    }
                    // END Parallax Animation


                    // Section HTML output
                    $html .= '<!-- Begin: ui/Theme-Builder Lite HTML output. https://www.ui-themebuilder.com/ -->';
                    $html .= '<' . $section_options['html_tag'] . $section_id . ' class="uk-section uk-position-relative' . $section_classes . '"' . $section_parallax . $scrollspy_animation_attr . $section_height_viewport . $srcset . $buildStyle . '>';

                    // SVG dividers
                    $html .= $section_divider_top;
                    $html .= $section_divider_bottom;

                    // visible section number
                    if ($section_key_show === "1" && isset($section_options['section_key_position'])) {
                        $add_key                = $key + 1;
                        $section_key_buildStyle = '';
                        // opacity style
                        if (isset($section_options['section_key_opacity']) && !empty($section_options['section_key_opacity'])) {
                            $section_key_style['opacity'] = $section_options['section_key_opacity'];
                        }
                        // font-size style
                        if (isset($section_options['section_key_font_size']) && !empty($section_options['section_key_font_size'])) {
                            $section_key_style['font-size'] = $section_options['section_key_font_size'] . '%';
                        }
                        // build inline styles
                        if (!empty($section_key_style)) {
                            $section_key_buildStyle = SitePageHelper::buildStyle($section_key_style);
                        }

                        $html .= '<div class="uk-visible@m">';
                        if ($key <= 8) {
                            $html .= '<div class="uk-position-large uk-position-' . $section_options['section_key_position'] . ' uk-text-bold ui-section-key" ' . $section_key_buildStyle . '>0' . $add_key . '</div>';
                        } else {
                            $html .= '<div class="uk-position-large uk-position-' . $section_options['section_key_position'] . ' uk-text-bold ui-section-key" ' . $section_key_buildStyle . '>' . $add_key . '</div>';
                        }
                        $html .= '</div>';
                    }

                    // Section Video Background
                    if ($custom_background === "5" && !empty($section_options['video_url'])) {
                        $html .= $video_source;
                    }

                    // Section Mask
                    if (isset($section_options['svg_mask_list']) && !empty($section_options['svg_mask_list'])) {
                        $html .= '<div class="uk-visible@m uk-position-absolute uk-width-1-1 uk-height-1-1 ' . $section_mask_classes . '"></div>';
                    }

                    // Section Shapes
                    if ($custom_background === "4" && !empty($section_options['shape'])) {
                        $html .= '<div class="uk-visible@m uk-position-absolute ' . $section_shape_class . $shape_shadow . $shape_background_image_position . '"></div>';
                    }

                    if (($section_full_width === "0") && !empty($section_options['container_width'])) {
                        $container_classes .= ' ' . $section_options['container_width'];
                        $html              .= '<div class="uk-container' . $container_classes . '"' . $container_buildStyle . '>';
                    }

                    if (($section_options['section_load_components'] === "0") && ($section_options['heading_alignment'] === 'uk-text-center') && ($section_options['section_height_viewport'] === "1")) {
                        if ((!empty($section_options['title']) || !empty($section_options['subtitle'])) && ($title_show === "1" && $section_options['use_svg_title'] === "0")) {
                            if (!empty($section_options['heading_alignment'])) {
                                $heading_alignment = ' ' . $section_options['heading_alignment'];
                            }
                            $html .= '<div class="' . trim($grid_gutter_class) . $position_relative . '" uk-grid="">';
                            $html .= '<div class="uk-width-1-1">';
                            $html .= '<div class="uk-position-center uk-flex uk-flex-center uk-flex-middle ' . $heading_alignment . '">';
                            $html .= '<div class="uk-margin-auto uk-margin-auto-vertical uk-width-auto">';
                            if (!empty($section_options['title'])) {
                                if (!empty($section_options['subtitle'])) {
                                    $title_class .= ' uk-margin-remove';
                                }
                                $html .= '<' . $section_options['title_heading'] . ' class="' . trim($title_class) . '"' . $parallax_attr_title . '>';
                                $html .= '<span>' . $section_options['title'] . '</span>';
                                $html .= '</' . $section_options['title_heading'] . '>';
                            }
                            if (!empty($section_options['subtitle'])) {
                                if (!empty($section_options['title'])) {
                                    $subtitleMargin = 'uk-margin-small-top';
                                }
                                $html .= '<' . $section_options['subtitle_heading'] . ' class="' . $subtitleMargin . $subtitle_class . '"' . $parallax_attr_subtitle . '>';
                                $html .= '<span>' . $section_options['subtitle'] . '</span>';
                                $html .= '</' . $section_options['subtitle_heading'] . '>';
                            }
                            $html .= '</div>';
                            $html .= '</div>';
                            $html .= '</div>';
                            $html .= '</div>';
                        } elseif ((!empty($section_options['title']) || !empty($section_options['subtitle'])) && ($title_show === "1" && $section_options['use_svg_title'] === "1")) {
                            // SVG Title/Subtitle Style -Viewport-height
                            // Title Stroke Settings
                            if ($section_options['svg_title_stroke'] === "1") {
                                if (!empty($section_options['svg_title_stroke_fill_color'])) {
                                    $fill_stroke = ' fill-stroke="' . $section_options['svg_title_stroke_fill_color'] . '"';
                                } else {
                                    $fill_stroke = '';
                                }
                                if (!empty($section_options['svg_title_stroke_color'])) {
                                    $stroke = ' stroke="' . $section_options['svg_title_stroke_color'] . '"';
                                } else {
                                    $stroke = '';
                                }
                                if (!empty($section_options['svg_title_stroke_width'])) {
                                    $stroke_width = ' stroke-width="' . $section_options['svg_title_stroke_width'] . '"';
                                } else {
                                    $stroke_width = '';
                                }

                                $title_stroke = $fill_stroke . $stroke . $stroke_width;
                            } else {
                                $title_stroke = '';
                            }

                            // Subtitle Stroke Settings
                            if ($section_options['svg_subtitle_stroke'] === "1") {
                                if (!empty($section_options['svg_subtitle_stroke_fill_color'])) {
                                    $sub_fill_stroke = ' fill-stroke="' . $section_options['svg_subtitle_stroke_fill_color'] . '"';
                                } else {
                                    $sub_fill_stroke = '';
                                }
                                if (!empty($section_options['svg_subtitle_stroke_color'])) {
                                    $sub_stroke = ' stroke="' . $section_options['svg_subtitle_stroke_color'] . '"';
                                } else {
                                    $sub_stroke = '';
                                }
                                if (!empty($section_options['svg_subtitle_stroke_width'])) {
                                    $sub_stroke_width = ' stroke-width="' . $section_options['svg_subtitle_stroke_width'] . '"';
                                } else {
                                    $sub_stroke_width = '';
                                }

                                $sub_title_stroke = $sub_fill_stroke . $sub_stroke . $sub_stroke_width;
                            } else {
                                $sub_title_stroke = '';
                            }
                            $html .= '<svg class="ui-svg-title uk-position-cover" width="100%" height="100%"' . $section_height_viewport . ' xmlns="http://www.w3.org/2000/svg">';
                            $html .= '<defs>';
                            $html .= '<mask id="text-mask-' . $key . '">';
                            $html .= '<rect width="100%" height="100%" fill="' . $section_options['mask_color'] . '" />';
                            $html .= '<g id="text-' . $key . '">';
                            if (!empty($section_options['title'])) {
                                $html .= '<text text-anchor="middle" x="' . $section_options['title_position_x'] . '%" y="' . $section_options['title_position_y'] . '%" id="svgTitle" class="' . trim($title_class) . '"' . $title_stroke . $parallax_attr_title . '>' . $section_options['title'] . '</text>';
                            }
                            if (!empty($section_options['subtitle'])) {
                                $html .= '<text text-anchor="middle" x="' . $section_options['subtitle_position_x'] . '%" y="' . $section_options['subtitle_position_y'] . '%" id="svgSubtitle" class="' . trim($subtitle_class) . '"' . $sub_title_stroke . $parallax_attr_subtitle . '>' . $section_options['subtitle'] . '</text>';
                            }
                            $html .= '</g>';
                            $html .= '</mask>';
                            $html .= '</defs>';
                            $html .= '<rect id="base" x="0" y="0" width="100%" height="100%" fill="' . $section_options['mask_color_output'] . '" mask="url(#text-mask-' . $key . ')" />';
                            $html .= '</svg>';
                        }
                    } elseif ((!empty($section_options['title']) || !empty($section_options['subtitle'])) && ($title_show === "1" && $section_options['use_svg_title'] === "0")) {
                        if (!empty($section_options['heading_alignment'])) {
                            $heading_alignment = ' ' . $section_options['heading_alignment'];
                        }
                        $html .= '<div class="' . trim($grid_gutter_class) . $position_relative . '" uk-grid="">';
                        $html .= '<div class="uk-width-1-1">';
                        $html .= '<div class="uk-padding-small' . $heading_alignment . '">';
                        if (!empty($section_options['title'])) {
                            if (!empty($section_options['subtitle'])) {
                                $title_class .= ' uk-margin-remove';
                            }
                            $html .= '<' . $section_options['title_heading'] . ' class="' . trim($title_class) . '"' . $parallax_attr_title . '>';
                            $html .= '<span>' . $section_options['title'] . '</span>';
                            $html .= '</' . $section_options['title_heading'] . '>';
                        }
                        if (!empty($section_options['subtitle'])) {
                            if (!empty($section_options['title'])) {
                                $subtitleMargin = 'uk-margin-small-top';
                            }
                            $html .= '<' . $section_options['subtitle_heading'] . ' class="' . $subtitleMargin . $subtitle_class . '"' . $parallax_attr_subtitle . '>';
                            $html .= '<span>' . $section_options['subtitle'] . '</span>';
                            $html .= '</' . $section_options['subtitle_heading'] . '>';
                        }
                        $html .= '</div>';
                        $html .= '</div>';
                        $html .= '</div>';
                    } elseif ((!empty($section_options['title']) || !empty($section_options['subtitle'])) && ($title_show === "1" && $section_options['use_svg_title'] === "1")) {
                        // SVG Title/Subtitle Style
                        // Title Stroke Settings
                        if ($section_options['svg_title_stroke'] === "1") {
                            if (!empty($section_options['svg_title_stroke_fill_color'])) {
                                $fill_stroke = ' fill-stroke="' . $section_options['svg_title_stroke_fill_color'] . '"';
                            } else {
                                $fill_stroke = '';
                            }
                            if (!empty($section_options['svg_title_stroke_color'])) {
                                $stroke = ' stroke="' . $section_options['svg_title_stroke_color'] . '"';
                            } else {
                                $stroke = '';
                            }
                            if (!empty($section_options['svg_title_stroke_width'])) {
                                $stroke_width = ' stroke-width="' . $section_options['svg_title_stroke_width'] . '"';
                            } else {
                                $stroke_width = '';
                            }

                            $title_stroke = $fill_stroke . $stroke . $stroke_width;
                        } else {
                            $title_stroke = '';
                        }

                        // Subtitle Stroke Settings
                        if ($section_options['svg_subtitle_stroke'] === "1") {
                            if (!empty($section_options['svg_subtitle_stroke_fill_color'])) {
                                $sub_fill_stroke = ' fill-stroke="' . $section_options['svg_subtitle_stroke_fill_color'] . '"';
                            } else {
                                $sub_fill_stroke = '';
                            }
                            if (!empty($section_options['svg_subtitle_stroke_color'])) {
                                $sub_stroke = ' stroke="' . $section_options['svg_subtitle_stroke_color'] . '"';
                            } else {
                                $sub_stroke = '';
                            }
                            if (!empty($section_options['svg_subtitle_stroke_width'])) {
                                $sub_stroke_width = ' stroke-width="' . $section_options['svg_subtitle_stroke_width'] . '"';
                            } else {
                                $sub_stroke_width = '';
                            }

                            $sub_title_stroke = $sub_fill_stroke . $sub_stroke . $sub_stroke_width;
                        } else {
                            $sub_title_stroke = '';
                        }

                        $html .= '<svg class="ui-svg-title uk-height-' . $section_options['mask_height'] . '" width="100%" height="100%" xmlns="http://www.w3.org/2000/svg">';
                        $html .= '<defs>';
                        $html .= '<mask id="text-mask-' . $key . '">';
                        $html .= '<rect width="100%" height="100%" fill="' . $section_options['mask_color'] . '" />';
                        $html .= '<g id="text-' . $key . '">';
                        if (!empty($section_options['title'])) {
                            $html .= '<text text-anchor="middle" x="' . $section_options['title_position_x'] . '%" y="' . $section_options['title_position_y'] . '%" id="svgTitle" class="' . trim($title_class) . '"' . $title_stroke . $parallax_attr_title . '>' . $section_options['title'] . '</text>';
                        }
                        if (!empty($section_options['subtitle'])) {
                            $html .= '<text text-anchor="middle" x="' . $section_options['subtitle_position_x'] . '%" y="' . $section_options['subtitle_position_y'] . '%" id="svgSubtitle" class="' . trim($subtitle_class) . '"' . $sub_title_stroke . $parallax_attr_subtitle . '>' . $section_options['subtitle'] . '</text>';
                        }
                        $html .= '</g>';
                        $html .= '</mask>';
                        $html .= '</defs>';
                        $html .= '<rect id="base" x="0" y="0" width="100%" height="100%" fill="' . $section_options['mask_color_output'] . '" mask="url(#text-mask-' . $key . ')" />';
                        $html .= '</svg>';
                    }
                    // END Title/Subtitle Settings

                    if ($section_options['section_load_components'] === "1") {
                        $columnClass = ($grid_gutter_class || $position_relative ? ' class="' . trim($grid_gutter_class) . $position_relative . '"': '');
                        $html .= '<div'.$columnClass.' uk-grid=""' . $scrollspy_animation_attr . $grid_match_attr . '>';

                        foreach ($section['columns'] as $column_index => $column) {
                            $column         = SitePageHelper::toArray($column);
                            $column_options = SitePageHelper::escapeHtmlArray($column['options']);
                            $title_show     = ($column_options['title_show'] ?? 1);

                            $column_class           = '';
                            $column_css             = '';
                            $column_bg_class        = '';
                            $columnPositionRelative = '';

                            $column_grid_gutter_class = '';
                            $column_heading_alignment = ' class="uk-padding-small uk-padding-remove-vertical"';

                            $flex_column_items_vertical = (!empty($column_options['flex_column_items_vertical']) ? ' ' . $column_options['flex_column_items_vertical'] : '');

                            $GridChildWidth = (!empty($column_options['grid_child_width']) ? ' uk-child-width-' . $column_options['grid_child_width'] . '@m' : ' uk-child-width-1-1');

                            $custom_column_background = (isset($column_options['custom_background']) && $column_options['custom_background'] ? $column_options['custom_background'] : "0");

                            $shape_position      = ($column_options['shape_position'] ?? 'left');
                            $shape_margin_top    = (!empty($column_options['shape_margin_top']) ? $column_options['shape_margin_top'] . 'px' : 0);
                            $shape_margin_right  = (!empty($column_options['shape_margin_right']) ? $column_options['shape_margin_right'] . 'px' : 0);
                            $shape_margin_bottom = (!empty($column_options['shape_margin_bottom']) ? $column_options['shape_margin_bottom'] . 'px' : 0);
                            $shape_margin_left   = (!empty($column_options['shape_margin_left']) ? $column_options['shape_margin_left'] . 'px' : 0);

                            $shape_position_zindex = ($column_options['shape_position_zindex'] ?? 0);
                            $shape_border          = (!empty($column_options['shape_border']) ? $column_options['shape_border'] . 'px' : '');
                            $shape_transform       = ($column_options['shape_transform'] ?? 'unset');
//                          $shape_shadow = ((isset($column_options['shape_shadow']) && !empty($column_options['shape_shadow'])) ? ' ' . $column_options['shape_shadow'] : '');

                            $shape_background_image_repeat     = ($column_options['shape_background_image_repeat'] ?? 0);
                            $shape_background_image_attachment = ($column_options['shape_background_image_attachment'] ?? "");
//                          $shape_background_image_position = ((isset($column_options['shape_background_image_position']) && !empty($column_options['shape_background_image_position']) && !empty($column_options['shape_background_image'])) ? ' ' . $column_options['shape_background_image_position'] : '');


                            if (!empty($column_options['id'])) {
                                $column_id = ' id="' . $column_options['id'] . '"';
                            } else {
                                $column_id = '';
                            }
                            if (!empty($column_options['class'])) {
                                $column_class .= ' ' . SitePageHelper::getClasses($column_options);
                            }

                            // Add Column Background
                            if ($custom_column_background === "1") {
                                // default column background color
                                if (!empty($column_options['background_color'])) {
                                    $column_css .= '.ui-column-' . $key . $column_index . '{';
                                    $column_css .= 'position: relative;';
                                    $column_css .= '}';

                                    if (!empty($flex_column_items_vertical)) {
                                        $column_css .= '.ui-column-' . $key . $column_index . ' > div > *::before{';
                                    } else {
                                        $column_css .= '.ui-column-' . $key . $column_index . ' > *::before{';
                                    }
                                    $column_css .= 'content: "";';
                                    $column_css .= 'position: absolute;';
                                    $column_css .= 'top: 0;';
                                    $column_css .= 'left: 40px;';
                                    $column_css .= 'width: calc(100% - 55px);';
                                    $column_css .= 'height: 100%;';
                                    $column_css .= 'z-index: -2;';
                                    $column_css .= 'background-color: ' . $column_options['background_color'] . ';';
                                    $column_css .= '}';

                                    $column_css .= '@media (min-width: 1200px){';

                                    if (!empty($flex_column_items_vertical)) {
                                        $column_css .= '.ui-column-' . $key . $column_index . ' > div > *::before{';
                                    } else {
                                        $column_css .= '.ui-column-' . $key . $column_index . ' > *::before{';
                                    }
                                    $column_css .= 'width: calc(100% - 40px);';
                                    $column_css .= '}';
                                    $column_css .= '}';

                                    $column_bg_class .= '';
                                }
                            } elseif ($custom_column_background === "2") {
                                $column_class           .= ' uk-position-relative';
                                $columnPositionRelative = ' uk-position-relative';

                                // square / circle shape
                                if (($column_options['shape'] === 'square' || $column_options['shape'] === 'circle') && ((!empty($column_options['shape_width']) && !empty($column_options['shape_height'])) || (!empty($column_options['shape_width_pixel']) && !empty($column_options['shape_height_pixel'])))) {
                                    // CSS
                                    $column_css .= '.ui-column-' . $key . $column_index . '::before{';

                                    $column_css .= 'content: " ";';
                                    $column_css .= 'position: absolute;';

                                    // width & height pixel or %
                                    if (!empty($column_options['shape_width']) && !empty($column_options['shape_height'])) {
                                        $column_css .= 'width: ' . $column_options['shape_width'] . '%;';
                                        $column_css .= 'height: ' . $column_options['shape_height'] . '%;';
                                    } elseif (!empty($column_options['shape_width_pixel']) && !empty($column_options['shape_height_pixel'])) {
                                        $column_css .= 'width: ' . $column_options['shape_width_pixel'] . 'px;';
                                        $column_css .= 'height: ' . $column_options['shape_height_pixel'] . 'px;';
                                    }
                                    // background-color
                                    if (!empty($column_options['shape_color'])) {
                                        $column_css .= 'background-color: ' . $column_options['shape_color'] . ';';
                                    } else {
                                        $column_css .= 'background-color: transparent;';
                                    }
                                    // background-image
                                    if (!empty($column_options['shape_background_image'])) {
                                        $column_css .= 'background-image: url("' . Uri::root() . $column_options['shape_background_image'] . '");';
                                        // image repeat
                                        if ($shape_background_image_repeat === "0") {
                                            $column_css .= 'background-repeat: no-repeat;';
                                        }
                                        // image position
                                        if (!empty($column_options['shape_background_image_position'])) {
                                            $backgroundPosition = '';

                                            if ($column_options['shape_background_image_position'] === 'uk-background-top-left') {
                                                $backgroundPosition = '0 0';
                                            } elseif ($column_options['shape_background_image_position'] === 'uk-background-top-center') {
                                                $backgroundPosition = '50% 0';
                                            } elseif ($column_options['shape_background_image_position'] === 'uk-background-top-right') {
                                                $backgroundPosition = '100% 0';
                                            } elseif ($column_options['shape_background_image_position'] === 'uk-background-center-left') {
                                                $backgroundPosition = '0 50%';
                                            } elseif ($column_options['shape_background_image_position'] === 'uk-background-center-center') {
                                                $backgroundPosition = '50% 50%';
                                            } elseif ($column_options['shape_background_image_position'] === 'uk-background-center-right') {
                                                $backgroundPosition = '100% 50%';
                                            } elseif ($column_options['shape_background_image_position'] === 'uk-background-bottom-left') {
                                                $backgroundPosition = '0 100%';
                                            } elseif ($column_options['shape_background_image_position'] === 'uk-background-bottom-center') {
                                                $backgroundPosition = '50% 100%';
                                            } elseif ($column_options['shape_background_image_position'] === 'uk-background-bottom-right') {
                                                $backgroundPosition = '100% 100%';
                                            }

                                            $column_css .= 'background-position: ' . $backgroundPosition . ';';
                                        }
                                        // image size
                                        if (!empty($column_options['shape_background_image_size'])) {
                                            $column_css .= 'background-size: ' . $column_options['shape_background_image_size'] . ';';
                                        }
                                        // image blend modes
                                        if (!empty($column_options['shape_background_image_blend_modes']) && !empty($column_options['shape_color'])) {
                                            $column_css .= 'background-blend-mode: ' . $column_options['shape_background_image_blend_modes'] . ';';
                                        }
                                        // image attachment
                                        if ($shape_background_image_attachment === "fixed") {
                                            $column_css .= 'background-attachment: fixed;';
                                        }
                                    }
                                    // position
                                    $column_css .= 'top: 0;';
                                    if ($shape_position === 'right') {
                                        $column_css .= 'right: 0;';
                                    } else {
                                        $column_css .= 'left: 0;';
                                    }
                                    // margin
                                    if (!$shape_margin_top || !$shape_margin_right || !$shape_margin_bottom || !$shape_margin_left) {
                                        $column_css .= 'margin: ' . $shape_margin_top . ' ' . $shape_margin_right . ' ' . $shape_margin_bottom . ' ' . $shape_margin_left . ';';
                                    }
                                    // border
                                    if ($shape_border && !empty($column_options['shape_border_style']) && !empty($column_options['shape_border_color'])) {
                                        $column_css .= 'border: ' . $shape_border . ' ' . $column_options['shape_border_style'] . ' ' . $column_options['shape_border_color'] . ';';
                                    }
                                    if ($column_options['shape'] === 'circle') {
                                        $column_css .= 'border-radius: 50%;';
                                    }
                                    // transform rotate
                                    if ($shape_transform === 'rotate' && !empty($column_options['shape_transform_rotate'])) {
                                        $column_css .= 'transform: rotate(' . $column_options['shape_transform_rotate'] . 'deg);';
                                        // transform rotate3d
                                    } elseif ($shape_transform === 'rotate3d' && !empty($column_options['shape_transform_rotate']) && !empty($column_options['shape_transform3d_1']) && !empty($column_options['shape_transform3d_2']) && !empty($column_options['shape_transform3d_3'])) {
                                        $column_css .= 'transform: rotate3d(' . $column_options['shape_transform3d_1'] . ', ' . $column_options['shape_transform3d_2'] . ', ' . $column_options['shape_transform3d_3'] . ', ' . $column_options['shape_transform_rotate'] . 'deg);';
                                    } else {
                                        $column_css .= 'transform: unset;';
                                    }
                                    // z-index
                                    if ($shape_position_zindex === "1") {
                                        $column_css .= 'z-index: 1;';
                                    }/* else {
                                            $column_css .= 'z-index: -1;';
                                        }*/
                                    $column_css .= '}';

                                    // triangle shape
                                } elseif ($column_options['shape'] === 'triangle' && !empty($column_options['shape_color'])) {
                                    // CSS
                                    $column_css .= '.ui-column-' . $key . $column_index . '::before{';
                                    // width & height
                                    $column_css .= 'width: 0;';
                                    $column_css .= 'height: 0;';

                                    // background-color
                                    $column_css .= 'background-color: transparent;';
                                    // position
                                    $column_css .= 'top: 0;';
                                    if ($shape_position === 'right') {
                                        $column_css .= 'right: 0;';
                                    } else {
                                        $column_css .= 'left: 0;';
                                    }
                                    // margin
                                    if (!$shape_margin_top || !$shape_margin_right || !$shape_margin_bottom || !$shape_margin_left) {
                                        $column_css .= 'margin: ' . $shape_margin_top . ' ' . $shape_margin_right . ' ' . $shape_margin_bottom . ' ' . $shape_margin_left . ';';
                                    }
                                    // border (build triangle)
                                    if (!empty($column_options['shape_height_pixel'])) {
                                        $column_css .= 'border-left: ' . $column_options['shape_height_pixel'] / 2 . 'px solid transparent;';
                                        $column_css .= 'border-right: ' . $column_options['shape_height_pixel'] / 2 . 'px solid transparent;';
                                        $column_css .= 'border-bottom: ' . $column_options['shape_height_pixel'] . 'px solid ' . $column_options['shape_color'] . ';';
                                    }
                                    // transform rotate
                                    if ($shape_transform === 'rotate' && !empty($column_options['shape_transform_rotate'])) {
                                        $column_css .= 'transform: rotate(' . $column_options['shape_transform_rotate'] . 'deg);';
                                        // transform rotate3d
                                    } elseif ($shape_transform === 'rotate3d' && !empty($column_options['shape_transform_rotate']) && !empty($column_options['shape_transform3d_1']) && !empty($column_options['shape_transform3d_2']) && !empty($column_options['shape_transform3d_3'])) {
                                        $column_css .= 'transform: rotate3d(' . $column_options['shape_transform3d_1'] . ', ' . $column_options['shape_transform3d_2'] . ', ' . $column_options['shape_transform3d_3'] . ', ' . $column_options['shape_transform_rotate'] . 'deg);';
                                    } else {
                                        $column_css .= 'transform: unset;';
                                    }
                                    // z-index
                                    if ($shape_position_zindex === "1") {
                                        $column_css .= 'z-index: 1;';
                                    }/* else {
                                            $column_css .= 'z-index: -1;';
                                        }*/
                                    $column_css .= '}';
                                }
                            }

                            if (!empty($column_options['color_style'])) {
                                $column_class .= ' ' . $column_options['color_style'];
                            }

                            // Add Inline CSS Background / Shapes
                            if (!empty($column_css)) {
                                $css_all .= preg_replace('/\s+/', ' ', $column_css);
                            }

                            // Column Grid Options
                            if (!empty($column_options['grid_gutter'])) {
                                $column_grid_gutter_class .= ' ' . $column_options['grid_gutter'];
                            }
                            if (!empty($column_options['grid_row_gap'])) {
                                $column_grid_gutter_class .= ' ' . $column_options['grid_row_gap'];
                            }
                            if (!empty($column_options['grid_divider'])) {
                                $column_grid_gutter_class .= ' ' . $column_options['grid_divider'];
                            }

                            // column html
                            $html .= '<div' . $column_id . ' class="ui-column ui-column-' . $key . $column_index . ' uk-width-' . $column['grid'] . '@m' . $column_class . $flex_column_items_vertical . '">';

                            if (!empty($flex_column_items_vertical)) {
                                $html .= '<div>';
                            }

                            // column title + subtitle html
                            if ((!empty($column_options['title']) || !empty($column_options['subtitle'])) && $title_show === "1") {
                                $titleAndSubtitleClassesColumn = SitePageHelper::getTitleAndSubtitleClasses($column_options);
                                $titleClassColumn              = ($titleAndSubtitleClassesColumn['title'] === '') ? '' : ' ' . $titleAndSubtitleClassesColumn['title'];
                                $subtitleClassColumn           = ($titleAndSubtitleClassesColumn['subtitle'] === '') ? '' : ' ' . $titleAndSubtitleClassesColumn['subtitle'];


                                if (isset($column_options['heading_alignment']) && !empty($column_options['heading_alignment'])) {
                                    $column_heading_alignment = ' class="uk-padding-small uk-padding-remove-vertical ' . $column_options['heading_alignment'] . '"';
                                }
                                $html .= '<div class="' . trim($column_grid_gutter_class . $columnPositionRelative) . '" uk-grid="">';
                                $html .= '<div class="uk-width-1-1">';
                                $html .= '<div' . $column_heading_alignment . '>';
                                if (!empty($column_options['title'])) {
                                    $html .= '<' . $column_options['title_heading'] . ' class="' . trim($titleClassColumn) . '">';
                                    $html .= '<span>' . $column_options['title'] . '</span>';
                                    $html .= '</' . $column_options['title_heading'] . '>';
                                }
                                if (!empty($column_options['subtitle'])) {
                                    $html .= '<' . $column_options['subtitle_heading'] . ' class="uk-margin-remove-top' . $subtitleClassColumn . '">';
                                    $html .= '<span>' . $column_options['subtitle'] . '</span>';
                                    $html .= '</' . $column_options['subtitle_heading'] . '>';
                                }
                                $html .= '</div>';
                                $html .= '</div>';
                                $html .= '</div>';
                            }

                            if (count($column['components'])) {
                                if ($column_options['masonry_grid'] === "1") {
                                    $child_width = '';
                                    if (!empty($column_options['masonry_grid_parallax'])) {
                                        $parallax = ' parallax:' . $column_options['masonry_grid_parallax'];
                                    } else {
                                        $parallax = '';
                                    }
                                    if (!empty($column_options['grid_m'])) {
                                        $child_width .= ' uk-child-width-' . $column_options['grid_m'] . '@m';
                                    }
                                    if (!empty($column_options['grid_l'])) {
                                        $child_width .= ' uk-child-width-' . $column_options['grid_l'] . '@l';
                                    }
                                    if (!empty($column_options['grid_xl'])) {
                                        $child_width .= ' uk-child-width-' . $column_options['grid_xl'] . '@xl';
                                    }
                                    $html .= '<div class="uk-child-width-1-1' . $child_width . $column_grid_gutter_class . $column_bg_class . '" uk-grid="masonry: true;' . $parallax . '">';
                                } else {
                                    $html .= '<div class="' . $GridChildWidth . $column_grid_gutter_class . $column_bg_class . '" uk-grid="">';
                                }

                                // Component output
                                if ($app->isClient('site')) {
                                    $html .= self::viewComponents($column['components'], '', '', '');
                                } else {
                                    $html .= self::viewComponents($column['components'], '', false, '');
                                }


                                $html .= '</div>';
                            }

                            if (!empty($flex_column_items_vertical)) {
                                $html .= '</div>';
                            }

                            $html .= '</div>';
                            // END column html
                        }
                        $html .= '</div>';
                    }

                    if ($section_full_width === "0") {
                        $html .= '</div>';
                    }

                    $html .= '</' . $section_options['html_tag'] . '>';
                    $html .= '<!-- End: ui/Theme-Builder Lite HTML output. -->';
                    // END section HTML
                }
                // END Section visibility and check date
            }

            // add Image Microdata to some components with images and alt attribute.
            $html = SitePageHelper::addImageMicrodata($html, ['ui-personal-card', 'ui-filtering']);

            // optimize CSS Output
            if (!empty($css_all)) {
                $optimizeCSS = str_replace(';}', '}', $css_all);
                $html    .= '<style>' . trim(str_replace(array ( "\r\n", "\r", "\n", '  '   ), '', $optimizeCSS)) . '</style>';
            }

            // generated cache files
            if ($cache === true && $id && $app->isClient('site')) {
                file_put_contents($cacheFile, '<?php die("Access Denied"); ?>' . trim(preg_replace('/^\s+|\n|\r|\s+$/m', '', $html)), LOCK_EX);

                // write index.html in cache folder if folder exists
                SitePageHelper::createIndexFile($cachePath);
            } elseif ($cache === false && file_exists($cacheFile)) {
                chmod($cacheFile, 0777);
                unlink($cacheFile);
            }

            // optimize HTML Output
            return trim(preg_replace('/^\s+|\n|\r|\s+$/m', '', $html));
            // return $html;
        }

        return false;
    }

    /**
     * Render Components.
     *
     * @param                $components
     * @param   string|null  $id
     * @param   bool         $cache
     * @param                $cacheTime
     *
     * @return string|bool
     *
     * @throws \Exception
     *
     * @since 1.2.8
     */
    public static function viewComponents($components, ?string $id, bool $cache, $cacheTime): string|bool
    {
        $app     = Factory::getApplication();
        $doc     = $app->getDocument();
        $wa      = $doc->getWebAssetManager();

        $cachePath = JPATH_ROOT . DIRECTORY_SEPARATOR . 'cache' . DIRECTORY_SEPARATOR . 'com_uithemebuilderlite' . DIRECTORY_SEPARATOR;  // use site cache folder, not administrator cache folder
        $cacheFile = $cachePath . $id . '.php';

        if ($id && $cache === true && $app->isClient('site') && !file_exists($cachePath)) {
            Folder::create($cachePath, 0755);
        }

        if ($cacheTime !== '0') {
            // Time for cache files: Module settings
            $time      = $cacheTime;
        } else {
            // Time for cache files: 60 × 60 × 24 = 24 Hours
            $time      = 60 * 60 * 24;
        }
        // Time for cache files: - $cacheTime
        if ($id && $cache === true && $app->isClient('site') && file_exists($cacheFile) && filemtime($cacheFile) > time() - $time) {
            $cacheContent = file_get_contents($cacheFile);
            // Remove the PHP tag from the cache content
            $cacheContent = str_replace('<?php die("Access Denied"); ?>', '', $cacheContent);

            $cacheCss            = '';
            $cacheScript         = '';
            $cacheScriptTypeJson = '';

            // Find all the <style> tags and extract its content.
            preg_match_all('/<style>(.*?)<\/style>/s', $cacheContent, $matchesStyles);
            if (!empty($matchesStyles[1])) {
                $cacheCss = implode("\n", $matchesStyles[1]);
            }

            // Find all the <script> tags and extract its content.
            preg_match_all('/<script>(.*?)<\/script>/s', $cacheContent, $matchesScripts);
            if (!empty($matchesScripts[1])) {
                $cacheScript = implode("\n", $matchesScripts[1]);
            }

            // Find all the <script type="application/ld+json"> tags and extract its content.
            preg_match_all('/<script\s+type="application\/ld\+json">(.*?)<\/script>/s', $cacheContent, $matchesJson);
            if (!empty($matchesJson[1])) {
                $cacheScriptTypeJson = implode(",", $matchesJson[1]);
            }

            // Add the extracted script as inline script
            if (!empty($cacheScript)) {
                $wa->addInlineScript($cacheScript, [], ['type' => 'module']);
            }

            // Add the extracted script as inline script
            if (!empty($cacheScriptTypeJson)) {
                $wa->addInlineScript($cacheScriptTypeJson, [], ['type' => 'application/ld+json']);
            }

            // Add the extracted CSS as inline style
            if (!empty($cacheCss)) {
                $wa->addInlineStyle($cacheCss);
            }

            // Return the modified content of the file
            return $cacheContent;
        }

        if (is_array($components)) {
            $html          = '';
            $cssComponent  = '';
            $component_css = '';

            foreach ($components as $item_index => $component) {
                $style             = [];
                $component         = SitePageHelper::toArray($component);
                $component_options = SitePageHelper::escapeHtmlArray($component['options']);
                $component_folder  = str_replace('_', '-', $component['type']);

                $component_css_class = 'ui-' . $component_folder . '-svg-bg-' . $item_index;
                $component_index     = 'ui-' . $component_folder . '-' . $item_index;
                $class               = 'ui-' . $component_folder . ' ' . $component_index;

                $class .= ' ' . SitePageHelper::getClasses($component_options);

                $custom_background             = ($component_options['custom_background'] ?? 0);
                $background_image_repeat       = ($component_options['background_image_repeat'] ?? 0);
                $parallax_animation_media      = isset($component_options['parallax_animation_media']) ? 'media:' . $component_options['parallax_animation_media'] . ';' : '';
                $parallax_animation_viewport   = isset($component_options['parallax_animation_viewport']) ? 'viewport:' . $component_options['parallax_animation_viewport'] . ';' : '';
                $parallax_animation_background = (!isset($component_options['parallax_animation_background']) || (($component_options['parallax_animation_background'] === "1"))) ? '1' : '0';
                $parallax_animation_box        = ($component_options['parallax_animation_box'] ?? 0);
                $scrollspy_animation           = ($component_options['scrollspy_animation'] ?? 0);
                $scrollspy_animation_repeat    = (!isset($component_options['scrollspy_animation_repeat']) || $component_options['scrollspy_animation_repeat'] === "1") ? '1' : '0';

                $component_background_image_srcset = ($component_options['background_image_srcset'] ?? 0);
                $component_srcset                  = '';
                $srcsets                           = [];
                $buildStyle                        = '';

                $currentDate_component      = Factory::getDate()->format('Y-m-d H:i:s');
                $get_publish_down_component = ($component_options['publish_down'] ?? null);
                $get_publish_up_component   = ($component_options['publish_up'] ?? null);

                $publish_down      = $get_publish_down_component == null ? null : $component_options['publish_down'];
                $isExpired         = !is_null($publish_down) && strtotime($publish_down) < strtotime($currentDate_component);
                $isNotPublishedYet = !is_null($publish_down) && strtotime($get_publish_up_component) > strtotime($currentDate_component);
                $isUnpublished     = $isNotPublishedYet || $isExpired;

                $component_show = true;

                if ($isExpired) {
                    $component_show = false;
                } elseif ($isUnpublished) {
                    $component_show = false;
                }

                if ($component_show) {
                    $html .= '<div>';

                        // Custom ID
                    if (!empty($component_options['id'])) {
                        $id = ' id="' . $component_options['id'] . '"';
                    } else {
                        $id = '';
                    }
                        // Scrollspy Animation
                    if ($scrollspy_animation === "1") {
                        $cls = '';

                        if (!empty($component_options['scrollspy_animation_type'])) {
                            $cls = 'cls: ' . $component_options['scrollspy_animation_type'] . ';';
                        }
                        if ($scrollspy_animation_repeat) {
                            $repeat = ' repeat: true';
                        } else {
                            $repeat = '';
                        }
                        if (!empty($component_options['scrollspy_animation_delay'])) {
                            $delay = ' delay: ' . $component_options['scrollspy_animation_delay'] . ';';
                        } else {
                            $delay = ' delay: 500;';
                        }

                        $scrollspy_animation_attr = ' uk-scrollspy="' . $cls . $delay . $repeat . '"';
                    } else {
                        $scrollspy_animation_attr = '';
                    }
                        // Parallax Animation
                    if ($parallax_animation_box === "1" || (!empty($parallax_animation_background) && ($custom_background === "1" && !empty($component_options['background_color'])))) {
                        if ((!empty($parallax_animation_media) && !empty($parallax_animation_background)) || ($parallax_animation_box === "1")) {
                            $parallax_attr_media = $parallax_animation_media;
                        } else {
                            $parallax_attr_media = '';
                        }

                        if ((!empty($parallax_animation_viewport) && !empty($parallax_animation_background)) || ($parallax_animation_box === "1")) {
                            $parallax_attr_viewport = $parallax_animation_viewport;
                        } else {
                            $parallax_attr_viewport = '';
                        }

                        if ($parallax_animation_box === "1") {
                            if (!empty($component_options['parallax_animation_box_opacity_values'])) {
                                $parallax_attr_box_opacity = 'opacity: ' . $component_options['parallax_animation_box_opacity_values'] . ';';
                            } else {
                                $parallax_attr_box_opacity = '';
                            }
                            if (!empty($component_options['parallax_animation_box_y_values'])) {
                                $parallax_attr_box_y = ' y: ' . $component_options['parallax_animation_box_y_values'] . ';';
                            } else {
                                $parallax_attr_box_y = '';
                            }
                            if (!empty($component_options['parallax_animation_box_x_values'])) {
                                $parallax_attr_box_x = ' x: ' . $component_options['parallax_animation_box_x_values'] . ';';
                            } else {
                                $parallax_attr_box_x = '';
                            }

                            if (!empty($component_options['parallax_animation_box_scale_values'])) {
                                $parallax_attr_box_scale = ' scale: ' . $component_options['parallax_animation_box_scale_values'] . ';';
                            } else {
                                $parallax_attr_box_scale = '';
                            }

                            if (!empty($component_options['parallax_animation_box_rotate_values'])) {
                                $parallax_attr_box_rotate = ' rotate: ' . $component_options['parallax_animation_box_rotate_values'] . ';';
                            } else {
                                $parallax_attr_box_rotate = '';
                            }
                        } else {
                            $parallax_attr_box_opacity = '';
                            $parallax_attr_box_scale   = '';
                            $parallax_attr_box_rotate  = '';
                            $parallax_attr_box_x       = '';
                            $parallax_attr_box_y       = '';
                        }

                        if (!empty($parallax_animation_background)) {
                            if (!empty($component_options['parallax_animation_background_x'])) {
                                $parallax_x = ' bgx: ' . $component_options['parallax_animation_background_x'] . ';';
                            } else {
                                $parallax_x = '';
                            }
                            if (!empty($component_options['parallax_animation_background_y'])) {
                                $parallax_y = ' bgy: ' . $component_options['parallax_animation_background_y'] . ';';
                            } else {
                                $parallax_y = '';
                            }
                            // Filters
                            if (!empty($component_options['parallax_animation_background_grayscale'])) {
                                $parallax_animation_background_grayscale = ' grayscale: ' . $component_options['parallax_animation_background_grayscale'] . ';';
                            } else {
                                $parallax_animation_background_grayscale = '';
                            }
                            if (!empty($component_options['parallax_animation_background_saturate'])) {
                                $parallax_animation_background_saturate = ' saturate: ' . $component_options['parallax_animation_background_saturate'] . ';';
                            } else {
                                $parallax_animation_background_saturate = '';
                            }
                            if (!empty($component_options['parallax_animation_background_color']) && !empty($component_options['background_color'])) {
                                $parallax_animation_background_color = ' background-color: ' . $component_options['background_color'] . ',' . $component_options['parallax_animation_background_color'] . ';';
                            } else {
                                $parallax_animation_background_color = '';
                            }
                        } else {
                            $parallax_x                              = '';
                            $parallax_y                              = '';
                            $parallax_animation_background_grayscale = '';
                            $parallax_animation_background_saturate  = '';
                            $parallax_animation_background_color     = '';
                        }

                        $parallax_attr_box = ' uk-parallax="' . $parallax_attr_box_opacity . $parallax_attr_box_y . $parallax_attr_box_x . $parallax_attr_box_scale . $parallax_attr_media . $parallax_attr_viewport . $parallax_attr_box_rotate . $parallax_x . $parallax_y . $parallax_animation_background_grayscale . $parallax_animation_background_saturate . $parallax_animation_background_color . '"';
                    } else {
                        $parallax_attr_box = '';
                    }

                        // Custom Background
                    if ($custom_background === "1") {
                        if ((!empty($component_options['background_image'])) || (!empty($component_options['background_image_size']))) {
                            if (!empty($component_options['background_image'])) {
                                // Background Srcset
                                if ($component_background_image_srcset === "1" && !empty($component_options['image_width_srcset'])) {
                                    [$width, $height] = getimagesize(Uri::root() . $component_options['background_image']);
                                    $ex        = '.' . pathinfo($component_options['background_image'], PATHINFO_EXTENSION);
                                    $file_name = 'th_' . pathinfo($component_options['background_image'], PATHINFO_FILENAME);
                                    // ratio
                                    $ratio = round($width / $height * 100);

                                    $typeItem         = str_replace('_', '-', $component['type']);
                                    $thumbnail_folder = 'media/com_uithemebuilderlite/ui-themebuilder-thumb/' . $typeItem;

                                    $tmp_srcset    = explode(',', $component_options['image_width_srcset']);
                                    $merge_srcsets = array_merge($srcsets, $tmp_srcset);
                                    $srcsets       = array_map('trim', $merge_srcsets);
                                    $srcsets       = array_unique($srcsets);

                                    $min_val = min($srcsets);

                                    $component_srcset .= ' data-sizes="';
                                    $component_srcset .= '(min-width:  ' . $min_val . 'px) 90vw, ';
                                    $component_srcset .= '(max-width: ' . $width . 'px) 100vw, ';
                                    $component_srcset .= '(max-aspect-ratio: ' . $width . '/' . $height . ') ' . $ratio . 'vh" ';
                                    $component_srcset .= 'data-srcset="';

                                    foreach ($srcsets as $i => $val) {
                                        if (is_numeric($val) && $val < 4000) {
                                            unset($srcsets[$i]);
                                            $component_srcset .= Uri::root() . $thumbnail_folder . '/' . $file_name . '_' . $val . $ex . ' ' . $val . 'w, ';
                                        }
                                    }

                                    $component_srcset .= Uri::root() . $component_options['background_image'] . ' ' . $width . 'w" ';
                                    $component_srcset .= 'data-src="' . Uri::root() . $component_options['background_image'] . '"';
                                    $component_srcset .= ' uk-img=""';
                                } else {
                                    $style['background-image'] = Uri::root() . $component_options['background_image'];
                                }

                                if ($background_image_repeat === "0") {
                                    $class .= ' uk-background-norepeat';
                                }

                                if (!empty($component_options['background_image_size'])) {
                                    $class .= ' ' . $component_options['background_image_size'];
                                }
                            }
                        }
                        if (!empty($component_options['background_color'])) {
                            $style['background-color'] = $component_options['background_color'];
                        }
                        if (((!empty($component_options['background_image'])) && (!empty($component_options['background_color']))) && !empty($component_options['background_image_blend_modes'])) {
                            $class .= ' ' . $component_options['background_image_blend_modes'];
                        }
                        if ((!empty($component_options['background_image'])) && (!empty($component_options['background_image_attachment']))) {
                            $class .= ' ' . $component_options['background_image_attachment'];
                        }
                        if ((!empty($component_options['background_image'])) && (!empty($component_options['background_image_position']))) {
                            $class .= ' ' . $component_options['background_image_position'];
                        }
                    }
                        // SVG Background
                    elseif ($custom_background === "2") {
                        if (!empty($component_options['svg_list'])) {
                            $svg_path                   = Uri::root() . $component_options['svg_list'];
                            $color_placeholder_1        = $component_options['svg_background_color_fallback'];
                            $color_placeholder_2        = $component_options['svg_background_color_2'];
                            $color_placeholder_3        = $component_options['svg_background_color_3'];
                            $text_placeholder           = $component_options['svg_background_text'];
                            $text_font_size_placeholder = $component_options['svg_background_text_font_size'] . 'px';
                            $opacity_placeholder        = $component_options['svg_background_image_opacity'];

                            $class .= ' ' . $component_css_class;

                            // CSS output
                            $component_css .= '.' . $component_css_class . '{';
                            if (!empty($component_options['svg_background_color_fallback'])) {
                                $component_css .= 'background: ' . $component_options['svg_background_color_fallback'] . ';';
                                $component_css .= 'background-color: ' . $component_options['svg_background_color_fallback'] . ';';
                            }
                            if (!empty($component_options['svg_background_attachment'])) {
                                $component_css .= 'background-attachment: ' . $component_options['svg_background_attachment'] . ';';
                            }
                            if (!empty($component_options['svg_background_image_size'])) {
                                $component_css .= 'background-size: ' . $component_options['svg_background_image_size'] . ';';
                            }
                            if (!empty($component_options['svg_background_image_repeat'])) {
                                $component_css .= 'background-repeat: ' . $component_options['svg_background_image_repeat'] . ';';
                            }
                            if (!empty($component_options['svg_background_position_x']) && !empty($component_options['svg_background_position_y'])) {
                                $component_css .= 'background-position: ' . $component_options['svg_background_position_x'] . ' ' . $component_options['svg_background_position_y'] . ';';
                            }
                            $component_css .= 'background-image: url("data:image/svg+xml,' . SitePageHelper::convertSVGtoURI($svg_path, $color_placeholder_1, $color_placeholder_2, $color_placeholder_3, $text_placeholder, $text_font_size_placeholder, $opacity_placeholder) . '");';
                            $component_css .= '}';

                            if (!empty($component_css)) {
                                $cssComponent .= preg_replace('/\s+/', ' ', $component_css);
                            }
                        }
                    }
                        // Gradient Background
                    elseif ($custom_background === "3") {
                        $color1 = ($component_options['gradient_color_1'] ?? '');
                        $color2 = ($component_options['gradient_color_2'] ?? '');
                        $color3 = ($component_options['gradient_color_3'] ?? '');

                        $color1_position = (isset($component_options['gradient_color_1_position']) ? $component_options['gradient_color_1_position'] . '%' : '');
                        $color2_position = (isset($component_options['gradient_color_2_position']) ? $component_options['gradient_color_2_position'] . '%' : '');
                        $color3_position = (isset($component_options['gradient_color_3_position']) ? $component_options['gradient_color_3_position'] . '%' : '');

                        $radial_conical_x_position = (isset($component_options['radial_conical_x_position']) ? $component_options['radial_conical_x_position'] . '%' : '');
                        $radial_conical_y_position = (isset($component_options['radial_conical_y_position']) ? $component_options['radial_conical_y_position'] . '%' : '');

                        // linear gradient
                        if ($component_options['gradient_effect'] === 'linear' && (!empty($color1) || !empty($color2) || !empty($color3))) {
                            $style['background-color'] = $color1; // Fallback color
                            // 3 active colors
                            if (!empty($color1) && !empty($color2) && !empty($color3)) {
                                $style['background'] = 'linear-gradient(' . $component_options['gradient_angle'] . 'deg, ' . $color1 . ' ' . $color1_position . ', ' . $color2 . ' ' . $color2_position . ', ' . $color3 . ' ' . $color3_position . ')';
                                // 2 active colors
                            } elseif (!empty($color1) && !empty($color2)) {
                                $style['background'] = 'linear-gradient(' . $component_options['gradient_angle'] . 'deg,' . $color1 . $color1_position . ', ' . $color2 . $color2_position . ')';
                            }
                            $style['background-attachment'] = 'scroll';

                            // radial gradient
                        } elseif ($component_options['gradient_effect'] === 'radial' && ((!empty($color1) && !empty($color2)) || !empty($color3)) && ($radial_conical_x_position && $radial_conical_y_position)) {
                            $style['background-color'] = $color1; // Fallback color
                            // 3 active colors
                            if (!empty($color1) && !empty($color2) && !empty($color3)) {
                                $style['background'] = 'radial-gradient(circle at ' . $radial_conical_x_position . ' ' . $radial_conical_y_position . ', ' . $color1 . ' ' . $color1_position . ', ' . $color2 . ' ' . $color2_position . ', ' . $color3 . ' ' . $color3_position . ')';
                                // 2 active colors
                            } elseif (!empty($color1) && !empty($color2)) {
                                $style['background'] = 'radial-gradient(circle at ' . $radial_conical_x_position . ' ' . $radial_conical_y_position . ', ' . $color1 . ' ' . $color1_position . ', ' . $color2 . ' ' . $color2_position . ')';
                            }
                            $style['background-attachment'] = 'scroll';

                            // conical gradient
                        } elseif ($component_options['gradient_effect'] === 'conical' && ((!empty($color1) && !empty($color2)) || !empty($color3)) && ($radial_conical_x_position && $radial_conical_y_position)) {
                            $style['background-color'] = $color1; // Fallback color
                            // 3 active colors
                            if (!empty($color1) && !empty($color2) && !empty($color3)) {
                                $style['background'] = 'conic-gradient(from ' . $component_options['gradient_angle'] . 'deg at ' . $radial_conical_x_position . ' ' . $radial_conical_y_position . ', ' . $color1 . ' ' . $color1_position . ', ' . $color2 . ' ' . $color2_position . ', ' . $color3 . ' ' . $color3_position . ')';
                                // 2 active colors
                            } elseif (!empty($color1) && !empty($color2)) {
                                $style['background'] = 'conic-gradient(from ' . $component_options['gradient_angle'] . 'deg at ' . $radial_conical_x_position . ' ' . $radial_conical_y_position . ', ' . $color1 . ' ' . $color1_position . ', ' . $color2 . ' ' . $color2_position . ')';
                            }
                        }

                        // Default Background
                    } elseif (!empty($component_options['background'])) {
                        $class .= ' ' . $component_options['background'];
                    }

                        // Component inline position style
                    if (!empty($component_options['component_position'])) {
                        $class .= ' ' . $component_options['component_position'];
                        // Left
                        if (!empty($component_options['component_position_left'])) {
                            $style['left'] = $component_options['component_position_left'];
                        }
                        // Right
                        if (!empty($component_options['component_position_right'])) {
                            $style['right'] = $component_options['component_position_right'];
                        }
                        // Top
                        if (!empty($component_options['component_position_top'])) {
                            $style['top'] = $component_options['component_position_top'];
                        }
                        // Bottom
                        if (!empty($component_options['component_position_bottom'])) {
                            $style['bottom'] = $component_options['component_position_bottom'];
                        }
                        // z-index
                        if (!empty($component_options['component_position_zindex'])) {
                            $style['z-index'] = $component_options['component_position_zindex'];
                        }
                    }
                        // set height in px or % (Image Pro)
                    if (!empty($component_options['component_image_height'])) {
                        $style['height'] = $component_options['component_image_height'];
                    }
                        // Module Style
                    if (!empty($style)) {
                        $buildStyle = SitePageHelper::buildStyle($style);
                    }

                        // unsplash.com search
                        $use_random_unsplash_search   = ($component_options['use_random_unsplash_search'] ?? 0);
                        $unsplash_search_query        = (isset($component_options['unsplash_search_query']) && !empty($component_options['unsplash_search_query']) ? $component_options['unsplash_search_query'] : '');
                        $unsplash_custom_search_query = (isset($component_options['unsplash_custom_search_query']) ? str_replace(' ', '', strtolower(trim($component_options['unsplash_custom_search_query']))) : '');
                        $unsplash_size                = (isset($component_options['unsplash_size']) && !empty($component_options['unsplash_size']) ? '&size=' . $component_options['unsplash_size'] : '');
                        $s_id                         = md5(uniqid($component_index . time(), true));

                        // add javascript
                    if (empty($component_options['image']) && $use_random_unsplash_search === "1" && !empty($unsplash_custom_search_query)) {
                        $search_query = str_replace(',', '_', $unsplash_custom_search_query);
                        $class        .= ' img-container-search-' . $search_query . $s_id;

                        $js = '
								window.displayImage' . $s_id . ' = function(data) {
									var searchDiv = document.querySelector(".img-container-search-' . $search_query . $s_id . '");
									var dataSrcElements = searchDiv.querySelectorAll("[data-src]");
									dataSrcElements.forEach(function(element) {
										var currentTime = new Date();
										var hours = currentTime.getHours();
										var minutes = currentTime.getMinutes();
										var seconds = currentTime.getSeconds();
										var time = hours + "" + minutes + "" + seconds;
										var dataSrc = "https://media.ui-themebuilder.com/com_uithemebuilderpro/unsplash/custom/" + data.fileName + "?" + time;
										element.setAttribute("data-src", dataSrc);
										
									});
								};
					
								window.addScript' . $search_query . $s_id . ' = function() {
									var script = document.createElement(\'script\');
									var head = document.getElementsByTagName(\'head\').item(0);
									var s = "?searchTerm=' . $unsplash_custom_search_query . '";
									var t = "&custom=true";
									var c = "&callback=displayImage' . $s_id . '";
									var z = "' . $unsplash_size . '";
									if (z) {
									var url =  "https://media.ui-themebuilder.com/com_uithemebuilderpro/unsplash/index.php" + s + t + c + z;
									} else {
									var url =  "https://media.ui-themebuilder.com/com_uithemebuilderpro/unsplash/index.php" + s + t + c;
									}
										
									script.setAttribute("src", url);
									script.setAttribute("type", "text/javascript");
									head.appendChild(script);
									
									if (script.parentElement==head) {
										head.removeChild(script);
									}
								};
								
								window.addEventListener(\'load\', function() {
									window.addScript' . $search_query . $s_id . '();
								});
							';
                    } elseif (empty($component_options['image']) && $use_random_unsplash_search === "1" && empty($unsplash_custom_search_query)) {
                        $class .= ' img-container-search-' . $s_id;

                        $js = '
								window.displayImage' . $s_id . ' = function(data) {
									var myDiv = document.querySelector(".img-container-search-' . $s_id . '");
									var dataSrcElements = myDiv.querySelectorAll("[data-src]");
									dataSrcElements.forEach(function(element) {
										var currentTime = new Date();
										var hours = currentTime.getHours();
										var minutes = currentTime.getMinutes();
										var seconds = currentTime.getSeconds();
										var time = hours + "" + minutes + "" + seconds;
										var dataSrc = "https://media.ui-themebuilder.com/com_uithemebuilderpro/unsplash/" + data.fileName + "?" + time;
										element.setAttribute("data-src", dataSrc);
									});
								};
					
								window.addScript' . $unsplash_search_query . $s_id . ' = function() {
									var script = document.createElement(\'script\');
									var head = document.getElementsByTagName(\'head\').item(0);
									var s = "?searchTerm=' . $unsplash_search_query . '";
									var t = "&custom=true";
									var c = "&callback=displayImage' . $s_id . '";
									var z = "' . $unsplash_size . '";
									if (z) {
									var url =  "https://media.ui-themebuilder.com/com_uithemebuilderpro/unsplash/index.php" + s + t + c + z;
									} else {
									var url =  "https://media.ui-themebuilder.com/com_uithemebuilderpro/unsplash/index.php" + s + t + c;
									}
								
									script.setAttribute("src", url);
									script.setAttribute("type", "text/javascript");
									head.appendChild(script);
									
									if (script.parentElement==head) {
										head.removeChild(script);
									}
								};
														
								window.addEventListener(\'load\', function() {
									window.addScript' . $unsplash_search_query . $s_id . '();
								});
							';
                    }

                    if (!empty($js) && $use_random_unsplash_search === "1") {
                        $html .= '<script>' . trim(preg_replace('/^\s+|\n|\r|\s+$/m', '', $js)) . '</script>';
                    }

                        // Get Component Layouts
                    if (File::exists(JPATH_THEMES . DIRECTORY_SEPARATOR . Factory::getApplication()->getTemplate() . DIRECTORY_SEPARATOR . 'html' . DIRECTORY_SEPARATOR . 'layouts' . DIRECTORY_SEPARATOR . 'com_uithemebuilderlite' . DIRECTORY_SEPARATOR . $component_folder . DIRECTORY_SEPARATOR . $component['type'] . '.php')) {
                        $layouts = new FileLayout($component['type'], JPATH_THEMES . DIRECTORY_SEPARATOR . Factory::getApplication()->getTemplate() . DIRECTORY_SEPARATOR . 'html' . DIRECTORY_SEPARATOR . 'layouts' . DIRECTORY_SEPARATOR . 'com_uithemebuilderlite' . DIRECTORY_SEPARATOR . $component_folder . DIRECTORY_SEPARATOR);
                    } else {
                        $layouts = new FileLayout($component['type'], JPATH_ROOT . DIRECTORY_SEPARATOR . 'components' . DIRECTORY_SEPARATOR . 'com_uithemebuilderlite' . DIRECTORY_SEPARATOR . 'layouts' . DIRECTORY_SEPARATOR . $component_folder . DIRECTORY_SEPARATOR);
                    }

                        $html .= '<div' . $id . ' class="' . trim($class) . '"' . $scrollspy_animation_attr . $parallax_attr_box . $component_srcset . $buildStyle . '>';

                            $html .= $layouts->render($component);

                        $html .= '</div>';

                    $html .= '</div>';
                }
            }

            // optimize CSS Output
            if (!empty($cssComponent)) {
                $optimizeCSS = str_replace(';}', '}', $cssComponent);
                $html    .= '<style>'.trim(str_replace(array ( "\r\n", "\r", "\n", '  ' ), '', $optimizeCSS)) . '</style>';
            }

            // Module Cache
            if ($cache === true && Factory::getApplication()->isClient('site')) {
                file_put_contents($cacheFile, '<?php die("Access Denied"); ?>' . trim(preg_replace('/^\s+|\n|\r|\s+$/m', '', $html)), LOCK_EX);

                // write index.html in cache folder if folder exists
                SitePageHelper::createIndexFile($cachePath);
            } elseif ($cache === false && file_exists($cacheFile)) {
                chmod($cacheFile, 0777);
                unlink($cacheFile);
            }

            // HTML Output
            return trim(preg_replace('/^\s+|\n|\r|\s+$/m', '', $html));
        }

        return false;
    }
}
