<?php
/**
 * @package      ui/Theme-Builder Lite
 * @author       Stephan W.
 * @author       url   https://www.ui-themebuilder.com/
 * @copyright    Copyright (C) 2021 ui-themebuilder.com, All rights reserved.
 * @developer    Stephan Wittling - https://www.ui-themebuilder.com/
 *               ui/Theme-Builder Lite is distributed in the hope that it will be useful,
 *               but WITHOUT ANY WARRANTY; without even the implied warranty of
 *               MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 *               See the GNU General Public License for more details.
 * @license      http://www.gnu.org/licenses/gpl.html GNU/GPL
 *********************************************************************************/

namespace SW\Component\uiThemeBuilderLite\Administrator\Table;

defined('_JEXEC') or die('Restricted access');

use Joomla\CMS\Application\ApplicationHelper;
use Joomla\CMS\Factory;
use Joomla\CMS\Filter\OutputFilter;
use Joomla\CMS\Language\Text;
use Joomla\CMS\Table\Table;
use Joomla\CMS\Versioning\VersionableTableInterface;
use Joomla\Database\DatabaseDriver;
use SW\Component\uiThemeBuilderLite\Administrator\Helper\PageHelper;

/**
 * ui/Theme-Builder Lite Table class.
 *
 * @since  v1.0.0
 */
class PageTable extends Table implements VersionableTableInterface
{

    /**
     * Constructor
     *
     * @param DatabaseDriver $db Database connector object
     *
     * @since   v1.0.0
     */
    public function __construct(DatabaseDriver $db)
    {
        $this->typeAlias = 'com_uithemebuilderlite.page';

        parent::__construct('#__uithemebuilderlite', 'id', $db);
    }

    /**
     * Stores a page.
     *
     * @param   boolean  $updateNulls  True to update fields even if they are null.
     *
     * @return  boolean  True on success, false on failure.
     *
     * @throws \Exception
     *
     * @since   v1.0.0
     */
    public function store($updateNulls = false): bool
    {
        // Transform the params field
        /*      if (is_array($this->params))
                {
                    $registry = new Registry($this->params);
                    $this->params = (string) $registry;
                } */

        $table = PageHelper::getTableInstance('PageTable', 'SW\\Component\\uiThemeBuilderLite\\Administrator\\Table\\');
        $date = Factory::getDate();
        $user = Factory::getApplication()->getIdentity();
        $alias = OutputFilter::stringURLSafe($this->alias);
        if ($this->id) {
            $this->modified = $date->toSql();
            $this->modified_by = $user->get('id');
        } else {
            if (!(int)$this->created) {
                $this->created = $date->toSql();
            }
            if (empty($this->created_by)) {
                $this->created_by = $user->get('id');
            }
        }
        if ($alias === '') {
            $alias = OutputFilter::stringURLSafe($this->title);
        }
        $this->alias = $alias;
        if ($table->load(array(
                'alias' => $alias
            )) && ($table->id != $this->id || $this->id == 0)) {
            // $this->setError(Text::_('COM_UITHEMEBUILDERLITE_ERROR_UNIQUE_ALIAS'));
            Factory::getApplication()->enqueueMessage(Text::_('COM_UITHEMEBUILDERLITE_ERROR_UNIQUE_ALIAS'), 'error');
            return false;
        }

        return parent::store($updateNulls);
    }

    /**
     * Generate a valid alias from title / date.
     * Remains public to be able to check for duplicated alias before saving
     *
     * @return  string
     *
     * @throws \Exception
     *
     * @since   v1.0.0
     */
    public function generateAlias(): string
    {
        if (empty($this->alias)) {
            $this->alias = $this->name;
        }

        $this->alias = ApplicationHelper::stringURLSafe($this->alias, $this->language);

        if (trim(str_replace('-', '', $this->alias)) == '') {
            $this->alias = Factory::getDate()->format('Y-m-d-H-i-s');
        }

        return $this->alias;
    }

    /**
     * Overloaded check function
     *
     * @return  boolean
     *
     * @throws \Exception
     * @see     Table::check
     *
     * @since   v1.0.0
     */
    public function check(): bool
    {
        try {
            parent::check();
        } catch (\Exception $e) {
            // $this->setError($e->getMessage());
            Factory::getApplication()->enqueueMessage($e->getMessage(), 'error');
            return false;
        }

        // Check the publish down date is not earlier than publish up.
        if ($this->publish_down > $this->_db->getNullDate() && $this->publish_down < $this->publish_up) {
            // $this->setError(Text::_('JGLOBAL_START_PUBLISH_AFTER_FINISH'));
            Factory::getApplication()->enqueueMessage(Text::_('JGLOBAL_START_PUBLISH_AFTER_FINISH'), 'error');
            return false;
        }

        // Set publish_up, publish_down to null if not set
        if (!$this->publish_up) {
            $this->publish_up = Factory::getDate()->format('Y-m-d-H-i-s');
        }

        if (!$this->publish_down) {
            $this->publish_down = '0000-00-00 00:00:00';
        }

        return true;
    }

    /**
     * @param $array
     * @param   string  $ignore
     *
     * @return bool
     *
     * @throws \Exception
     *
     * @since   v1.0.0
     */
    public function bind($array, $ignore = ''): bool
    {
        if (isset($array['id'])) {
            $date = Factory::getDate();
            $user = Factory::getApplication()->getIdentity();
            if ($array['id']) {
                $array['modified'] = $date->toSql();
                $array['modified_by'] = $user->get('id');
            } else {
                if (!(int)$array['created']) {
                    $array['created'] = $date->toSql();
                }
                if (empty($array['created_by'])) {
                    $array['created_by'] = $user->get('id');
                }
            }
        }

        // Bind the rules
        if (isset($array['rules']) && is_array($array['rules'])) {
            $this->setRules($array['rules']);
        }

        return parent::bind($array, $ignore);
    }

    /**
     * Get the type alias for the history table
     *
     * @return  string  The alias as described above
     *
     * @since   v1.0.0
     */
    public function getTypeAlias(): string
    {
        return $this->typeAlias;
    }

    /**
     *
     * @return string
     *
     * @since   v1.0.0
     */
    protected function _getAssetTitle(): string
    {
        return $this->title;
    }

    /**
     *
     * @return string
     *
     * @since   v1.0.0
     */
    protected function _getAssetName(): string
    {
        $k = $this->_tbl_key;
        return 'com_uithemebuilderlite.page.' . (int)$this->$k;
    }

    /**
     * @param   \Joomla\CMS\Table\Table|null  $table
     * @param                                 $id
     *
     * @return int
     *
     * @since   v1.0.0
     */
    protected function _getAssetParentId(Table $table = null, $id = null): int
    {
        $asset = PageHelper::getTableInstance('Asset');
        $asset->loadByName('com_uithemebuilderlite');
        return $asset->id;
    }
}
