<?php
/**
 * @package      ui/Theme-Builder Lite
 * @author       Stephan W.
 * @author       url   https://www.ui-themebuilder.com/
 * @copyright    Copyright (C) 2023 ui-themebuilder.com, All rights reserved.
 * @developer    Stephan Wittling - https://www.ui-themebuilder.com/
 *               ui/Theme-Builder Lite is distributed in the hope that it will be useful,
 *               but WITHOUT ANY WARRANTY; without even the implied warranty of
 *               MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 *               See the GNU General Public License for more details.
 * @license      http://www.gnu.org/licenses/gpl.html GNU/GPL
 *********************************************************************************/

namespace SW\Component\uiThemeBuilderLite\Administrator\Service\HTML;

\defined('_JEXEC') or die;


use Joomla\CMS\Factory;
use Joomla\CMS\Language\Associations;
use Joomla\CMS\Language\Text;
use Joomla\CMS\Layout\LayoutHelper;
use Joomla\CMS\Router\Route;
use Joomla\Utilities\ArrayHelper;

/**
 * Page HTML class.
 *
 * @since  1.0.0
 */
class AdministratorService
{
	/**
	 * Get the associated language flags
	 *
	 * @param   integer  $pageid  The item id to search associations
	 *
	 * @return  string  The language HTML
	 *
	 * @throws \Exception
	 *
	 * @since   v1.0.0
	 */
	public function association(int $pageid): string
	{
		// Defaults
		$html = '';

		// Get the associations
		if ($associations = Associations::getAssociations('com_uithemebuilderlite', '#__uithemebuilderlite', 'com_uithemebuilderlite.item', $pageid, 'id', null)) {
			foreach ($associations as $tag => $associated) {
				$associations[$tag] = (int)$associated->id;
			}

			// Get the associated page items
			$db    = Factory::getContainer()->get('DatabaseDriver');
			$query = $db->getQuery(true)
				->select('c.id, c.title as title')
				->select('l.sef as lang_sef, lang_code')
				->from('#__uithemebuilderlite as c')
				->select('cat.title as category_title')
				->join('LEFT', '#__categories as cat ON cat.id=c.catid')
				->where('c.id IN (' . implode(',', array_values($associations)) . ')')
				->where('c.id != ' . $pageid)
				->join('LEFT', '#__languages as l ON c.language=l.lang_code')
				->select('l.image')
				->select('l.title as language_title');
			$db->setQuery($query);

			try {
				$items = $db->loadObjectList('id');
			} catch (\RuntimeException $e) {
				throw new \Exception($e->getMessage(), 500, $e);
			}

			if ($items)
			{
				foreach ($items as &$item)
				{
					$text    = strtoupper($item->lang_sef);
					$url     = Route::_('index.php?option=com_uithemebuilderlite&task=page.edit&id=' . (int) $item->id);
					$tooltip = '<strong>' . htmlspecialchars($item->language_title, ENT_QUOTES, 'UTF-8') . '</strong><br>'
						. htmlspecialchars($item->title, ENT_QUOTES, 'UTF-8') . '<br>' . Text::sprintf('JCATEGORY_SPRINTF', $item->category_title);
					$classes = 'badge badge-secondary';

					$item->link = '<a href="' . $url . '" title="' . $item->language_title . '" class="' . $classes . '">' . $text . '</a>'
						. '<div role="tooltip" id="tip' . (int) $item->id . '">' . $tooltip . '</div>';
				}
			}

			$html = LayoutHelper::render('joomla.content.associations', $items);
		}

		return $html;
	}

	/**
	 * Show the featured/not-featured icon.
	 *
	 * @param   integer  $value      The featured value.
	 * @param   integer  $i          Id of the item.
	 * @param   boolean  $canChange  Whether the value can be changed or not.
	 *
	 * @return  string    The anchor tag to toggle featured/unfeatured pages.
	 *
	 * @since   v1.0.0
	 */
	public function featured(int $value, int $i, bool $canChange = true): string
	{
		// Array of image, task, title, action
		$states = [
			0 => ['unfeatured', 'pages.featured', 'COM_UITHEMEBUILDERLITE_UNFEATURED', 'JGLOBAL_TOGGLE_FEATURED'],
			1 => ['featured', 'pages.unfeatured', 'JFEATURED', 'JGLOBAL_TOGGLE_FEATURED'],
		];
		$state = ArrayHelper::getValue($states, (int)$value, $states[1]);
		$icon = $state[0];

		if ($canChange) {
			$html = '<a href="#" onclick="return Joomla.listItemTask(\'cb' . $i . '\',\'' . $state[1] . '\')" class="tbody-icon'
				. ($value == 1 ? ' active' : '') . '" title="' . Text::_($state[3])
				. '"><span class="icon-' . $icon . '" aria-hidden="true"></span></a>';
		} else {
			$html = '<a class="tbody-icon disabled' . ($value == 1 ? ' active' : '')
				. '" title="' . Text::_($state[2]) . '"><span class="icon-' . $icon . '" aria-hidden="true"></span></a>';
		}

		return $html;
	}
}
