<?php
/**
 * @package      ui/Theme-Builder Lite
 * @author       Stephan W.
 * @author       url   https://www.ui-themebuilder.com/
 * @copyright    Copyright (C) 2021 ui-themebuilder.com, All rights reserved.
 * @developer    Stephan Wittling - https://www.ui-themebuilder.com/
 *               ui/Theme-Builder Lite is distributed in the hope that it will be useful,
 *               but WITHOUT ANY WARRANTY; without even the implied warranty of
 *               MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 *               See the GNU General Public License for more details.
 * @license      http://www.gnu.org/licenses/gpl.html GNU/GPL
 *********************************************************************************/

namespace SW\Component\uiThemeBuilderLite\Administrator\Model;

defined('_JEXEC') or die;

use Joomla\CMS\Factory;
use Joomla\CMS\Filter\OutputFilter;
use Joomla\CMS\Language\Associations;
use Joomla\CMS\Language\LanguageHelper;
use Joomla\CMS\Language\Text;
use Joomla\CMS\MVC\Model\AdminModel;
use Joomla\CMS\MVC\Model\WorkflowBehaviorTrait;
use Joomla\CMS\MVC\Model\WorkflowModelInterface;
use Joomla\CMS\Table\Table;
use Joomla\CMS\Versioning\VersionableModelTrait;
use Joomla\Component\Categories\Administrator\Helper\CategoriesHelper;
use Joomla\Database\ParameterType;
use Joomla\Utilities\ArrayHelper;
use SW\Component\uiThemeBuilderLite\Administrator\Helper\PageHelper;

/**
 * Item Model for a Page.
 *
 * @since  v0.0.1
 */
class PageModel extends AdminModel implements WorkflowModelInterface
{

    use WorkflowBehaviorTrait, VersionableModelTrait;

    /**
     * The type alias for this content type.
     *
     * @var    string
     *
     * @since  v1.0.0
     */
    public $typeAlias = 'com_uithemebuilderlite.page';

    /**
     * @var object item
     *
     * @since  v1.0.0
     */
    protected $item;

    /**
     * The context used for the associations table
     *
     * @var    string
     *
     * @since  v1.0.0
     */
    protected $associationsContext = 'com_uithemebuilderlite.item';

    /**
     * Batch copy/move command. If set to false, the batch copy/move command is not supported
     *
     * @var  string
     *
     * @since  v1.0.0
     */
    protected $batch_copymove = 'category_id';

    /**
     * Allowed batch commands
     *
     * @var array
     *
     * @since  v1.0.0
     */
    protected $batch_commands = array(
        'assetgroup_id' => 'batchAccess',
        'language_id' => 'batchLanguage',
        'user_id' => 'batchUser',
    );

    /**
     * Method to get the row form.
     *
     * @param   array    $data      Data for the form.
     * @param   boolean  $loadData  True if the form is to load its own data (default case), false if not.
     *
     * @return  \JForm|boolean  A \JForm object on success, false on failure
     *
     * @throws \Exception
     * @since   v1.0.0
     */
    public function getForm($data = array(), $loadData = true): \JForm|bool
    {
        $form = $this->loadForm('com_uithemebuilderlite.page', 'page', array('control' => 'jform', 'load_data' => $loadData));

        if ($form === null) {
            return false;
        }
        $jinput = Factory::getApplication()->input;
        $id = $jinput->get('id', 0);

        if ($this->getState('page.id')) {
            $id = $this->getState('page.id');
            $form->setFieldAttribute('catid', 'action', 'core.edit');
            $form->setFieldAttribute('catid', 'action', 'core.edit.own');
        } else {
            $form->setFieldAttribute('catid', 'action', 'core.create');
        }

        $user = Factory::getApplication()->getIdentity();
        if (($id != 0 && (!$user->authorise('core.edit.state', 'com_uithemebuilderlite.page.' . (int) $id))) || ($id == 0 && !$user->authorise('core.edit.state', 'com_uithemebuilderlite'))) {
            $form->setFieldAttribute('published', 'disabled', 'true');
            $form->setFieldAttribute('published', 'filter', 'unset');
        }

        return $form;
    }

    /**
     * Method to toggle the featured setting of pages.
     *
     * @param   array    $pks    The ids of the items to toggle.
     * @param   integer  $value  The value to toggle to.
     *
     * @return  boolean  True on success.
     *
     * @throws \Exception
     *
     * @since   v1.0.0
     */
    public function featured(array $pks, int $value = 0): bool
    {
        // Sanitize the ids.
        $pks = ArrayHelper::toInteger((array)$pks);

        if (empty($pks)) {
            $this->setError(Text::_('COM_UITHEMEBUILDERLITE_NO_ITEM_SELECTED'));

            return false;
        }

        $table = $this->getTable();

        try {
            $db = Factory::getContainer()->get('DatabaseDriver');

            $query = $db->getQuery(true);
            $query->update($db->quoteName('#__uithemebuilderlite'));
            $query->set($db->quoteName('featured') . ' = :featured');
            $query->whereIn($db->quoteName('id'), $pks);
            $query->bind(':featured', $value, ParameterType::INTEGER);

            $db->setQuery($query);

            $db->execute();
        } catch (\Exception $e) {
            Factory::getApplication()->enqueueMessage($e->getMessage(), 'error');

            return false;
        }

        $table->reorder();

        // Clean component's cache
        $this->cleanCache();

        return true;
    }

    /**
     * @param   string  $type
     * @param   string  $prefix
     * @param $config
     *
     * @return bool|\Joomla\CMS\Table\Table
     *
     * @since   v1.0.0
     */
    public function getTable($type = 'PageTable', $prefix = 'SW\\Component\\uiThemeBuilderLite\\Administrator\\Table\\', $config = array()): Table|bool
    {
        return PageHelper::getTableInstance($type, $prefix, $config);
    }

    /**
     * Method to save the form data.
     *
     * @param   array  $data  The form data.
     *
     * @return  boolean  True on success.
     *
     * @throws \Exception
     *
     * @since   v1.0.0
     */
    public function save($data): bool
    {
        $app = Factory::getApplication();

        if ($app->input->get('task') == 'save2copy') {
            $table = $this->getTable();
            $table->load(array('alias' => $data['alias']));

            if ($data['title'] == $table->title) {
                [$title, $alias] = $this->generateNewTitle(null, $data['alias'], $data['title']);
                $data['title'] = $title;
                $data['alias'] = $alias;
            } elseif ($data['alias'] == $table->alias) {
                $data['alias'] = '';
            }
            $data['published'] = 0;
            $data['hits'] = '0';
            $data['id'] = 0;
            $task = 'apply';
        }


        // Alter the title for save as copy
        /*if ($app->input->get('task') == 'save2copy') {
            $origTable = $this->getTable();

            if ($app->isClient('site')) {
                $origTable->load($app->input->get('a_id'));

                if ($origTable->title === $data['title']) {
                    //
                     // If title of article is not changed, set alias to original page alias so that Joomla! will generate
                     // new Title and Alias for the copied page
                     //
                    $data['alias'] = $origTable->alias;
                } else {
                    $data['alias'] = '';
                }
            } else {
                $origTable->load($app->input->get('a_id'));
            }

            if ($data['title'] == $origTable->title) {
                [$title, $alias] = $this->generateNewTitle($data['catid'], $data['alias'], $data['title']);
                $data['title'] = $title;
                $data['alias'] = $alias;
            } elseif ($data['alias'] == $origTable->alias) {
                $data['alias'] = '';
            }
        }*/

        // Automatic handling of alias for empty fields
        if (in_array($app->input->get('task'), array('apply', 'save', 'save2new')) && (!isset($data['id']) || (int) $data['id'] == 0) && $data['alias'] == null) {
            if ($app->get('unicodeslugs') == 1) {
                $data['alias'] = OutputFilter::stringUrlUnicodeSlug($data['title']);
            } else {
                $data['alias'] = OutputFilter::stringURLSafe($data['title']);
            }

            $table = $this->getTable();

            if ($table->load(array('alias' => $data['alias'], 'catid' => $data['catid']))) {
                $msg = Text::_('COM_UITHEMEBUILDERLITE_ERROR_UNIQUE_ALIAS');
            }

            [$title, $alias] = $this->generateNewTitle($data['catid'], $data['alias'], $data['title']);
            $data['alias'] = $alias;

            if (isset($msg)) {
                $app->enqueueMessage($msg, 'warning');
            }
        }

        // Create a new category, if needed.
        $createCategory = true;

        if (is_null($data['catid'])) {
            // When there is no catid passed don't try to create one
            $createCategory = false;
        }

        // If category ID is provided, check if it's valid.
        if (is_numeric($data['catid']) && $data['catid']) {
            $createCategory = !CategoriesHelper::validateCategoryId($data['catid'], 'com_uithemebuilderlite');
        }

        // Save New Category
        if ($createCategory && $this->canCreateCategory()) {
            $category = [
                // Remove #new# prefix, if exists.
                'title' => str_starts_with($data['catid'], '#new#') ? substr($data['catid'], 5) : $data['catid'],
                'parent_id' => 1,
                'extension' => 'com_uithemebuilderlite',
                'language' => $data['language'],
                'published' => 1,
            ];

            /** @var \Joomla\Component\Categories\Administrator\Model\CategoryModel $categoryModel */
            $categoryModel = Factory::getApplication()->bootComponent('com_categories')
                ->getMVCFactory()->createModel('Category', 'Administrator', ['ignore_request' => true]);

            // Create a new category.
            if (!$categoryModel->save($category)) {
                Factory::getApplication()->enqueueMessage($categoryModel->getMessage(), 'error');

                return false;
            }

            // Get the Category ID.
            $data['catid'] = $categoryModel->getState('category.id');
        }


        $data['created_by'] = $this->checkExistingUser($data['created_by']);
        parent::save($data);

        return true;
    }

    /**
     * @param $id
     *
     * @return int|null
     *
     * @throws \Exception
     *
     * @since   v1.0.0
     */
    protected function checkExistingUser($id): ?int
    {
        $currentUser = Factory::getApplication()->getIdentity();
        $user_id = $currentUser->id;
        if ($id) {
            $user = Factory::getApplication()->getIdentity();
            if ($user->id) {
                $user_id = $id;
            }
        }
        return $user_id;
    }

    /**
     * Method to get the data that should be injected in the form.
     *
     * @return  mixed  The data for the form.
     *
     * @throws \Exception
     *
     * @since   v1.0.0
     */
    protected function loadFormData(): mixed
    {
        $data = $this->getItem();

        $this->preprocessData('com_uithemebuilderlite.page', $data);

        return $data;
    }

    /**
     * Method to get a single record.
     *
     * @param   null  $pk  The id of the primary key.
     *
     * @return  mixed  Object on success, false on failure.
     *
     * @throws \Exception
     *
     * @since   v1.0.0
     */
    public function getItem($pk = null): mixed
    {
        $item = parent::getItem($pk);

        // Load associated page items
        $assoc = Associations::isEnabled();

        if ($assoc) {
            $item->associations = array();

            if ($item->id != null) {
                $associations = Associations::getAssociations('com_uithemebuilderlite', '#__uithemebuilderlite', 'com_uithemebuilderlite.item', $item->id, 'id', null);

                foreach ($associations as $tag => $association) {
                    $item->associations[$tag] = $association->id;
                }
            }
        }

        return $item;
    }

    /**
     * Preprocess the form.
     *
     * @param   \JForm  $form   Form object.
     * @param   object  $data   Data object.
     * @param   string  $group  Group name.
     *
     * @return  void
     *
     * @throws \Exception
     *
     * @since   v1.0.0
     */
    protected function preprocessForm(\JForm $form, $data, $group = 'content'): void
    {
        if ($this->canCreateCategory()) {
            $form->setFieldAttribute('catid', 'allowAdd', 'true');

            // Add a prefix for categories created on the fly.
            $form->setFieldAttribute('catid', 'customPrefix', '#new#');
        }

        if (Associations::isEnabled()) {
            $languages = LanguageHelper::getContentLanguages(false, true, null, 'ordering', 'asc');

            if (count($languages) > 1) {
                $addform = new \SimpleXMLElement('<form />');
                $fields = $addform->addChild('fields');
                $fields->addAttribute('name', 'associations');
                $fieldset = $fields->addChild('fieldset');
                $fieldset->addAttribute('name', 'item_associations');

                foreach ($languages as $language) {
                    $field = $fieldset->addChild('field');
                    $field->addAttribute('name', $language->lang_code);
                    $field->addAttribute('type', 'modal_page');
                    $field->addAttribute('language', $language->lang_code);
                    $field->addAttribute('label', $language->title);
                    $field->addAttribute('translate_label', 'false');
                    $field->addAttribute('select', 'true');
                    $field->addAttribute('new', 'true');
                    $field->addAttribute('edit', 'true');
                    $field->addAttribute('clear', 'true');
                }

                $form->load($addform, false);
            }
        }

        parent::preprocessForm($form, $data, $group);
    }

    /**
     * Is the user allowed to create an on the fly category?
     *
     * @return  boolean
     *
     * @throws \Exception
     *
     * @since   1.0.2
     */
    private function canCreateCategory(): bool
    {
        return Factory::getApplication()->getIdentity()->authorise('core.create', 'com_uithemebuilderlite');
    }

    /**
     * Prepare and sanitise the table before saving.
     *
     * @param \Joomla\CMS\Table\Table $table The Table object
     *
     * @return  void
     *
     * @since   v1.0.0
     */
    protected function prepareTable($table): void
    {
        // Increment the content version number.
        $table->version++;
    }

    /**
     * Build an SQL query to load the list data.
     *
     * @return  \JDatabaseQuery
     *
     * @since   v1.0.0
     */
    protected function getListQuery(): \JDatabaseQuery
    {
        // Create a new query object.
        $db = Factory::getContainer()->get('DatabaseDriver');
        $query = $db->getQuery(true);
        // Select the required fields from the table.
        $query->select(
            $db->quoteName(['a.id', 'a.title', 'a.alias', 'a.access', 'a.catid', 'a.published', 'a.publish_up', 'a.publish_down'])
        );
        $query->from($db->quoteName('#__uithemebuilderlite', 'a'));
        // Join over the asset groups.
        $query->select($db->quoteName('ag.title', 'access_level'))
            ->join(
                'LEFT',
                $db->quoteName('#__viewlevels', 'ag') . ' ON ' . $db->quoteName('ag.id') . ' = ' . $db->quoteName('a.access')
            );
        // Join over the categories.
        $query->select($db->quoteName('c.title', 'category_title'))
            ->join(
                'LEFT',
                $db->quoteName('#__categories', 'c') . ' ON ' . $db->quoteName('c.id') . ' = ' . $db->quoteName('a.catid')
            );

        return $query;
    }

    /**
     *
     *
     * @param   null  $group
     * @param   int   $client_id
     *
     * @return void
     *
     * @since   v1.0.0
     */
    protected function cleanCache($group = null, $client_id = 0): void
    {
        parent::cleanCache('com_uithemebuilderlite');
    }
}
