<?php
/**
 * @package      ui/Theme-Builder Lite
 * @author       Stephan W.
 * @author       url   https://www.ui-themebuilder.com/
 * @copyright    Copyright (C) 2021 ui-themebuilder.com, All rights reserved.
 * @developer    Stephan Wittling - https://www.ui-themebuilder.com/
 *               ui/Theme-Builder Lite is distributed in the hope that it will be useful,
 *               but WITHOUT ANY WARRANTY; without even the implied warranty of
 *               MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 *               See the GNU General Public License for more details.
 * @license      http://www.gnu.org/licenses/gpl.html GNU/GPL
 *********************************************************************************/

namespace SW\Component\uiThemeBuilderLite\Administrator\Helper;

defined('_JEXEC') or die('Restricted access');

use Joomla\CMS\Association\AssociationExtensionHelper;
use Joomla\CMS\Language\Associations;
use Joomla\CMS\Table\Table;
use SW\Component\uiThemeBuilderLite\Site\Helper\AssociationHelper;
use SW\Component\uiThemeBuilderLite\Administrator\Helper\PageHelper;

/**
 * Content associations helper.
 *
 * @since  v1.0.0
 */
class AssociationsHelper extends AssociationExtensionHelper
{
    /**
     * The extension name
     *
     * @var     array $extension
     *
     * @since   v1.0.0
     */
    protected $extension = 'com_uithemebuilderlite';

    /**
     * Array of item types
     *
     * @var     array $itemTypes
     *
     * @since   v1.0.0
     */
    protected $itemTypes = array('page', 'category');

    /**
     * Has the extension association support
     *
     * @var     boolean $associationsSupport
     *
     * @since   v1.0.0
     */
    protected $associationsSupport = true;

    /**
     * Method to get the associations for a given item.
     *
     * @param   integer  $id    Id of the item
     * @param   string   $view  Name of the view
     *
     * @return  array   Array of associations for the item
     *
     * @throws \Exception
     *
     * @since  v1.0.0
     */
    public function getAssociationsForItem($id = 0, $view = null): array
    {
        return AssociationHelper::getAssociations($id, $view);
    }

    /**
     * Get the associated items for an item
     *
     * @param   string  $typeName  The item type
     * @param   int     $id        The id of item for which we need the associated items
     *
     * @return  array
     *
     * @throws \Exception
     *
     * @since   v1.0.0
     */
    public function getAssociations(string $typeName, int $id): array
    {
        $type = $this->getType($typeName);

        $context = $this->extension . '.item';
        $catidField = 'catid';

        if ($typeName === 'category') {
            $context = 'com_categories.item';
            $catidField = '';
        }

        // Get the associations.
        $associations = Associations::getAssociations(
            $this->extension,
            $type['tables']['a'],
            $context,
            $id,
            'id',
            'alias',
            $catidField
        );

        return $associations;
    }

    /**
     * Get information about the type
     *
     * @param string $typeName The item type
     *
     * @return  array  Array of item types
     *
     * @since   v1.0.0
     */
    public function getType($typeName = ''): array
    {
        $fields = $this->getFieldsTemplate();
        $tables = array();
        $joins = array();
        $support = $this->getSupportTemplate();
        $title = '';

        if (in_array($typeName, $this->itemTypes)) {
            switch ($typeName) {
                case 'page':
                    $fields['title'] = 'a.title';
                    $fields['state'] = 'a.published';

                    $support['state'] = true;
                    $support['acl'] = true;
                    $support['category'] = true;
                    $support['save2copy'] = true;

                    $tables = array(
                        'a' => '#__uithemebuilderlite'
                    );

                    $title = 'page';
                    break;

                case 'category':
                    $fields['created_user_id'] = 'a.created_user_id';
                    $fields['ordering'] = 'a.lft';
                    $fields['level'] = 'a.level';
                    $fields['catid'] = '';
                    $fields['state'] = 'a.published';

                    $support['state'] = true;
                    $support['acl'] = true;
                    $support['checkout'] = false;
                    $support['level'] = false;

                    $tables = array(
                        'a' => '#__categories'
                    );

                    $title = 'category';
                    break;
            }
        }

        return array(
            'fields' => $fields,
            'support' => $support,
            'tables' => $tables,
            'joins' => $joins,
            'title' => $title
        );
    }

    /**
     * Get default values for fields array
     *
     * @return  array
     *
     * @since   v1.0.0
     */
    protected function getFieldsTemplate(): array
    {
        return array(
            'id' => 'a.id',
            'title' => 'a.title',
            'alias' => 'a.alias',
            'ordering' => '',
            'menutype' => '',
            'level' => '',
            'catid' => 'a.catid',
            'language' => 'a.language',
            'access' => 'a.access',
            'state' => 'a.state',
            'created_user_id' => '',
            'checked_out' => '',
            'checked_out_time' => ''
        );
    }

    /**
     * Get item information
     *
     * @param   string  $typeName  The item type
     * @param   int     $id        The id of item for which we need the associated items
     *
     * @return  Table|null
     *
     * @since   v1.0.0
     */
    public function getItem(string $typeName, int $id): ?Table
    {
        if (empty($id)) {
            return null;
        }

        $table = null;

        switch ($typeName) {
            case 'page':
                $table = PageHelper::getTableInstance('PageTable', 'SW\\Component\\uiThemeBuilderLite\\Administrator\\Table\\');
                break;

            case 'category':
                $table = PageHelper::getTableInstance('Category');
                break;
        }

        if (empty($table)) {
            return null;
        }

        $table->load($id);

        return $table;
    }
}
