<?php
/**
 * @package      ui/Theme-Builder Lite
 * @author       Stephan W.
 * @author       url   https://www.ui-themebuilder.com/
 * @copyright    Copyright (C) 2021 ui-themebuilder.com, All rights reserved.
 * @developer    Stephan Wittling - https://www.ui-themebuilder.com/
 *               ui/Theme-Builder Lite is distributed in the hope that it will be useful,
 *               but WITHOUT ANY WARRANTY; without even the implied warranty of
 *               MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 *               See the GNU General Public License for more details.
 * @license      http://www.gnu.org/licenses/gpl.html GNU/GPL
 *********************************************************************************/

namespace SW\Component\uiThemeBuilderLite\Administrator\Field\Modal;

defined('JPATH_BASE') or die;

use Joomla\CMS\Factory;
use Joomla\CMS\Form\FormField;
use Joomla\CMS\HTML\HTMLHelper;
use Joomla\CMS\Language\LanguageHelper;
use Joomla\CMS\Language\Text;
use Joomla\CMS\Session\Session;
use Joomla\Database\ParameterType;

/**
 * Supports a modal page picker.
 *
 * @since  v0.0.1
 */
class PageField extends FormField
{

	/**
	 * The form field type.
	 *
	 * @var    string
	 *
	 * @since  v0.0.1
	 */
	protected $type = 'Modal_Page';

	/**
	 * Method to get the field input markup.
	 *
	 * @return  string  The field input markup.
	 *
	 * @throws \Exception
	 *
	 * @since  v0.0.1
	 */
	protected function getInput()
	{
		$allowNew = ((string)$this->element['new'] === 'true');
		$allowEdit = ((string)$this->element['edit'] === 'true');
		$allowClear = ((string)$this->element['clear'] !== 'false');
		$allowSelect = ((string)$this->element['select'] !== 'false');
		$allowPropagate = ((string)$this->element['propagate'] === 'true');

		$languages = LanguageHelper::getContentLanguages(array(0, 1), false);

		// Load language
		Factory::getApplication()->getLanguage()->load('com_uithemebuilderlite', JPATH_ADMINISTRATOR);

		// The active page id field.
		$value = (int)$this->value ?: '';

		// Create the modal id.
		$modalId = 'Page_' . $this->id;

		/** @var \Joomla\CMS\WebAsset\WebAssetManager $wa */
		$wa = Factory::getApplication()->getDocument()->getWebAssetManager();

		// Add the modal field script to the document head.
		$wa->useScript('keepalive')
			->useScript('form.validate')
			->useScript('field.modal-fields');

		// Script to proxy the select modal function to the modal-fields.js file.
		if ($allowSelect) {
			static $scriptSelect = null;

			if (is_null($scriptSelect)) {
				$scriptSelect = array();
			}

			if (!isset($scriptSelect[$this->id])) {
				$wa->addInlineScript("
				window.jSelectPage_" . $this->id . " = function (id, title, language) {
					window.processModalSelect('Page', '" . $this->id . "', id, title, language);
				}", [], ['type' => 'module']);

				Text::script('JGLOBAL_ASSOCIATIONS_PROPAGATE_FAILED');

				$scriptSelect[$this->id] = true;
			}
		}

		// Setup variables for display.
		$linkPages = 'index.php?option=com_uithemebuilderlite&amp;view=pages&amp;layout=modal&amp;tmpl=component&amp;' . Session::getFormToken() . '=1';
		$linkPage = 'index.php?option=com_uithemebuilderlite';

		if (isset($this->element['language'])) {
			$linkPages .= '&amp;forcedLanguage=' . $this->element['language'];
			$linkPage .= '&amp;forcedLanguage=' . $this->element['language'];
			$modalTitle = Text::_('COM_UITHEMEBUILDERLITE_SELECT_PAGE') . ' &#8212; ' . $this->element['label'];
		} else {
			$modalTitle = Text::_('COM_UITHEMEBUILDERLITE_SELECT_PAGE');
		}

		$urlSelect = $linkPages . '&amp;function=jSelectPage_' . $this->id;
		$urlEdit = $linkPage . '&task=page.edit&id=';
		$urlNew = $linkPage . '&amp;task=page.add';

		if ($value) {
			$db    = Factory::getContainer()->get('DatabaseDriver');
			$query = $db->getQuery(true)
				->select($db->quoteName('title'))
				->from($db->quoteName('#__uithemebuilderlite'))
				->where($db->quoteName('id') . ' = :value')
				->bind(':value', $value, ParameterType::INTEGER);
			$db->setQuery($query);

			try {
				$title = $db->loadResult();
			} catch (\RuntimeException $e) {
				Factory::getApplication()->enqueueMessage($e->getMessage(), 'error');
			}
		}

		$title = empty($title) ? Text::_('COM_UITHEMEBUILDERLITE_SELECT_PAGE') : htmlspecialchars($title, ENT_QUOTES, 'UTF-8');

		// The current page display field.
		$html = '';

		if ($allowSelect || $allowNew || $allowEdit || $allowClear) {
			$html .= '<span class="input-group">';
		}

		$html .= '<input class="form-control" id="' . $this->id . '_name" type="text" value="' . $title . '" readonly size="35">';

		$document = Factory::getApplication()->getDocument();
		$document->addStyleDeclaration('#' . $this->id . '_new:before{content:"";}');

		$wa->addInlineScript('
			document.getElementById("' . $this->id . '_edit").addEventListener("click", function(){
			var id = document.getElementById("' . $this->id . '_id").value;
			window.open("' . $urlEdit . '" + id); })', [], ['type' => 'module']);

		// Select page button
		if ($allowSelect) {
			$html .= '<button'
				. ' class="btn btn-primary' . ($value ? ' hidden' : '') . '"'
				. ' id="' . $this->id . '_select"'
				. ' data-bs-toggle="modal"'
				. ' type="button"'
				. ' data-bs-target="#ModalSelect' . $modalId . '">'
				. '<span class="icon-file" aria-hidden="true"></span> ' . Text::_('JSELECT')
				. '</button>';
		}

		// New page button
		if ($allowNew) {
			$html .= '<a'
				. ' class="btn btn-secondary' . ($value ? ' hidden' : '') . '"'
				. ' id="' . $this->id . '_new"'
				. ' href="' . $urlNew . '"'
				. ' target="_blank">'
				. '<span class="fas fa-plus" aria-hidden="true"></span> ' . Text::_('JACTION_CREATE')
				. '</a>';
		}

		// Edit page button
		if ($allowEdit) {
			$html .= '<a'
				. ' class="btn btn-secondary' . ($value ? '' : ' hidden') . '"'
				. ' id="' . $this->id . '_edit"'
				. ' href="javascript:void(0);"'
				. '>'
				. '<span class="fas fa-pen-square" aria-hidden="true"></span> ' . Text::_('JACTION_EDIT')
				. '</a>';
		}

		// Clear page button
		if ($allowClear) {
			$html .= '<button'
				. ' class="btn btn-secondary' . ($value ? '' : ' hidden') . '"'
				. ' id="' . $this->id . '_clear"'
				. ' type="button"'
				. ' onclick="window.processModalParent(\'' . $this->id . '\'); return false;">'
				. '<span class="icon-times" aria-hidden="true"></span> ' . Text::_('JCLEAR')
				. '</button>';
		}

		// Propagate page button
		if ($allowPropagate && count($languages) > 2) {
			// Strip off language tag at the end
			$tagLength = (int)strlen($this->element['language']);
			$callbackFunctionStem = substr("jSelectPage_" . $this->id, 0, -$tagLength);

			$html .= '<button'
				. ' class="btn btn-primary' . ($value ? '' : ' hidden') . '"'
				. ' type="button"'
				. ' id="' . $this->id . '_propagate"'
				. ' title="' . Text::_('JGLOBAL_ASSOCIATIONS_PROPAGATE_TIP') . '"'
				. ' onclick="Joomla.propagateAssociation(\'' . $this->id . '\', \'' . $callbackFunctionStem . '\');">'
				. '<span class="icon-sync" aria-hidden="true"></span> ' . Text::_('JGLOBAL_ASSOCIATIONS_PROPAGATE_BUTTON')
				. '</button>';
		}

		if ($allowSelect || $allowNew || $allowEdit || $allowClear) {
			$html .= '</span>';
		}

		// Select page modal
		if ($allowSelect) {
			$html .= HTMLHelper::_(
				'bootstrap.renderModal',
				'ModalSelect' . $modalId,
				array(
					'title' => $modalTitle,
					'url' => $urlSelect,
					'height' => '400px',
					'width' => '800px',
					'bodyHeight' => 70,
					'modalWidth' => 80,
					'footer' => '<button type="button" class="btn btn-secondary" data-bs-dismiss="modal">'
						. Text::_('JLIB_HTML_BEHAVIOR_CLOSE') . '</button>',
				)
			);
		}

		// Note: class='required' for client side validation.
		$class = $this->required ? ' class="required modal-value"' : '';

		$html .= '<input type="hidden" id="' . $this->id . '_id" ' . $class . ' data-required="' . (int)$this->required . '" name="' . $this->name
			. '" data-text="' . htmlspecialchars(Text::_('COM_UITHEMEBUILDERLITE_SELECT_PAGE'), ENT_COMPAT, 'UTF-8') . '" value="' . $value . '">';

		return $html;
	}

	/**
	 * Method to get the field label markup.
	 *
	 * @return  string  The field label markup.
	 *
	 * @since   v0.0.1
	 */
	protected function getLabel(): string
	{
		return str_replace($this->id, $this->id . '_name', parent::getLabel());
	}

}
