<?php
/**
 * @package      ui/Theme-Builder Lite
 * @author       Stephan W.
 * @author       url   https://www.ui-themebuilder.com/
 * @copyright    Copyright (C) 2023 ui-themebuilder.com, All rights reserved.
 * @developer    Stephan Wittling - https://www.ui-themebuilder.com/
 *               ui/Theme-Builder Lite is distributed in the hope that it will be useful,
 *               but WITHOUT ANY WARRANTY; without even the implied warranty of
 *               MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 *               See the GNU General Public License for more details.
 * @license      http://www.gnu.org/licenses/gpl.html GNU/GPL
 *********************************************************************************/

namespace SW\Component\uiThemeBuilderLite\Administrator\Controller;

\defined('_JEXEC') or die;

use Joomla\CMS\Application\CMSApplication;
use Joomla\CMS\Factory;
use Joomla\CMS\Language\Text;
use Joomla\CMS\MVC\Controller\AdminController;
use Joomla\CMS\MVC\Factory\MVCFactoryInterface;
use Joomla\Input\Input;
use Joomla\Utilities\ArrayHelper;
use SW\Component\uiThemeBuilderLite\Administrator\Helper\PageHelper;

/**
 * ui/Theme-Builder Lite list controller class.
 *
 * @since  v1.0.0
 */
class PagesController extends AdminController
{
    /**
     * Constructor.
     *
     * @param   array                                             $config   An optional associative array of configuration settings.
     *                                                                      Recognized key values include 'name', 'default_task', 'model_path', and
     *                                                                      'view_path' (this list is not meant to be comprehensive).
     * @param   \Joomla\CMS\MVC\Factory\MVCFactoryInterface|null  $factory  The factory.
     * @param   null                                              $app      The JApplication for the dispatcher
     * @param   null                                              $input    Input
     *
     * @throws \Exception
     * @since   v1.0.0
     */
    public function __construct($config = array(), MVCFactoryInterface $factory = null, $app = null, $input = null)
    {
        parent::__construct($config, $factory, $app, $input);

        $this->registerTask('unfeatured', 'featured');
    }

    /**
     * Method to toggle the featured setting of a list of foos.
     *
     * @return  void
     *
     * @since   v1.0.0
     */
    public function featured(): void
    {
        // Check for request forgeries
        $this->checkToken();

        $ids = $this->input->get('cid', array(), 'array');
        $values = array('featured' => 1, 'unfeatured' => 0);
        $task = $this->getTask();
        $value = ArrayHelper::getValue($values, $task, 0, 'int');

        $model = $this->getModel();

        // Access checks.
        foreach ($ids as $i => $id) {
            $item = $model->getItem($id);

            if (!$this->app->getIdentity()->authorise('core.edit.state', 'com_uithemebuilderlite.category.' . (int)$item->catid)) {
                // Prune items that you can't change.
                unset($ids[$i]);
                $this->app->enqueueMessage(Text::_('JLIB_APPLICATION_ERROR_EDITSTATE_NOT_PERMITTED'), 'notice');
            }
        }

        if (empty($ids)) {
            $this->app->enqueueMessage(Text::_('COM_UITHEMEBUILDERLITE_NO_ITEM_SELECTED'), 'warning');
        } else {
            // Publish the items.
            if (!$model->featured($ids, $value)) {
                $this->app->enqueueMessage($model->getError(), 'warning');
            }

            if ($value == 1) {
                $message = Text::plural('COM_UITHEMEBUILDERLITE_N_ITEMS_FEATURED', count($ids));
            } else {
                $message = Text::plural('COM_UITHEMEBUILDERLITE_N_ITEMS_UNFEATURED', count($ids));
            }
        }

        $this->setRedirect('index.php?option=com_uithemebuilderlite&view=pages', $message);
    }

    /**
     * Proxy for getModel.
     *
     * @param string $name   The name of the model.
     * @param string $prefix The prefix for the PHP class name.
     * @param array  $config Array of configuration parameters.
     *
     * @return  \Joomla\CMS\MVC\Model\BaseDatabaseModel
     *
     * @since   v1.0.0
     */
    public function getModel($name = 'Page', $prefix = 'Administrator', $config = array('ignore_request' => true))
    {
        return parent::getModel($name, $prefix, $config);
    }

    /**
     *
     * @return void
     *
     * @throws \Exception
     * @since v0.0.1
     */
    public function import(): void
    {
        $app = Factory::getApplication();
        if ($importClass = PageHelper::getProParams('import')) {
            $importClass->importFile();
        } else {
            $msg = Text::_('COM_UITHEMEBUILDERLITE_PRO');
            $app->redirect("index.php?option=com_uithemebuilderlite&view=pages", $msg);
        }
    }

    /**
     *
     * @return void
     *
     * @throws \Exception
     *
     * @since v0.0.1
     */
    public function export(): void
    {
        $app = Factory::getApplication();
        if ($exportClass = PageHelper::getProParams('export')) {
            $exportClass->exportFile();
        } else {
            $msg = Text::_('COM_UITHEMEBUILDERLITE_PRO');
            $app->redirect("index.php?option=com_uithemebuilderlite&view=pages", $msg);
        }
    }

    /**
     *
     * @return void
     *
     * @throws \Exception
     * @since v0.0.1
     */
    public function save(): void
    {
        $app = Factory::getApplication();
        if ($saveClass = PageHelper::getProParams('save')) {
            $saveClass->exportFile();
        } else {
            $msg = Text::_('COM_UITHEMEBUILDERLITE_PRO');
            $app->redirect("index.php?option=com_uithemebuilderlite&view=pages", $msg);
        }
    }
}
